/***********************************************************************
*
* dis - a tape image.
*
***********************************************************************/

#include <stdio.h>
#include <string.h>

#include "ubcd.h"

#define MAXRECSIZE 300000 /* The size of a tape record */

static int recmode;
static int colbin;
static int recnum;
static int filenum;
static int altchars;
static int absimg;
static int tapformat;
static int raw;
static int image;
static int ibsys;

/* Symbol tables */
typedef struct _opcode {
	int	opbase;
	char	*name;
	char	type;
} t_opcode;

typedef struct _sym {
	int	value;
	char	*name;
} t_sym;

#define TYPE_A	0		/* Basic opcode format */
#define TYPE_B  1		/* OpcodeF Lo[,T] */
#define TYPE_C  2		/* Opcode  Y,C */
#define TYPE_D  3		/* Opcode Y[,T] */
#define TYPE_E	4		/* Opcode */
#define TYPE_F	5		/* Opcode T*/
#define TYPE_G	6		/* Opcode K*/
#define TYPE_P	8		/* Positive 0760 instructs */
#define TYPE_N	9		/* Negitive 0760 instructs */
#define TYPE_X  10
#define TYPE_I	11		/* I/O opcodes */

/* Opcodes */
#define TXI	1
#define TIX	2
#define TXH	3
#define STR	5
#define TNX	6
#define TXL	7

/* Positive opcodes */
#define HTR	0000
#define TRA	0020
#define TTR	0021
#define RICA	0022
#define RICC	0024
#define RICE	0026
#define RICG	0027
#define TEFA	0030
#define TEFC	0031
#define TEFE	0032
#define TEFG	0033
#define TLQ	0040
#define IIA	0041
#define TIO	0042
#define OAI	0043
#define PAI	0044
#define TIF	0046
#define IIR	0051
#define RFT	0054
#define SIR	0055
#define RNT	0056
#define RIR	0057
#define TCOA	0060
#define TCOB	0061
#define TCOC	0062
#define TCOD	0063
#define TCOE	0064
#define TCOF	0065
#define TCOG	0066
#define TCOH	0067
#define TSX	0074
#define TZE	0100
#define TIA	0101
#define CVR	0114
#define TPL	0120
#define XCA	0131
#define TOV	0140
#define TQP	0162
#define TQO	0161
#define MPY	0200
#define VLM	0204
#define DVH	0220
#define DVP	0221
#define VDH	0224
#define VDP	0225
#define FDH	0240
#define FDP	0241
#define FMP	0260
#define DFMP	0261
#define FAD	0300
#define DFAD	0301
#define FSB	0302
#define DFSB	0303
#define FAM	0304
#define DFAM	0305
#define FSM	0306
#define DFSM	0307
#define ANS	0320
#define ERA	0322
#define CAS	0340
#define ACL	0361
#define HPR	0420
#define OSI	0442
#define ADD	0400
#define ADM	0401
#define SUB	0402
#define IIS	0440
#define LDI	0441
#define DLD	0443
#define OFT	0444
#define RIS	0445
#define ONT	0446
#define LDA	0460	/* 704 only */
#define CLA	0500
#define CLS	0502
#define ZET	0520
#define XEC	0522
#define LXA	0534
#define LAC	0535
#define RSCA	0540
#define RSCC	0541
#define RSCE	0542
#define RSCG	0543
#define STCA	0544
#define STCC	0545
#define STCE	0546
#define STCG	0547
#define LDQ	0560
#define ENB	0564
#define STZ	0600
#define STO	0601
#define SLW	0602
#define STI	0604
#define STA	0621
#define STD	0622
#define STT	0625
#define STP	0630
#define SXA	0634
#define SCA	0636
#define SCHA	0640
#define SCHC	0641
#define SCHE	0642
#define SCHG	0643
#define SCDA	0644
#define SCDC	0645
#define SCDE	0646
#define SCDG	0647
#define CPY	0700	/* 704 only */
#define PAX	0734
#define PAC	0737
#define PXA	0754
#define PCA	0756
#define NOP	0761
#define RDS	0762
#define BSR	0764
#define LLS	0763
#define LRS	0765
#define WRS	0766
#define ALS	0767
#define WEF	0770
#define ARS	0771
#define REW	0772
#define AXT	0774
#define DSR     0775
#define SDN	0776
 
/* Negative opcodes */
#define ESNT	04021
#define RICB	04022
#define RICD	04024
#define RICF	04026
#define RICH	04027
#define TEFB	04030
#define TEFD	04031
#define TEFF	04032
#define TEFH	04033
#define RIA	04042
#define PIA	04046
#define IIL	04051
#define LFT	04054
#define SIL	04055
#define LNT	04056
#define RIL	04057
#define TCNA	04060
#define TCNB	04061
#define TCNC	04062
#define TCND	04063
#define TCNE	04064
#define TCNF	04065
#define TCNG	04066
#define TCNH	04067
#define TNZ	04100
#define TIB	04101
#define CAQ	04114
#define TMI	04120
#define XCL	04130
#define TNO	04140
#define CRQ	04154
#define DUFA	04301
#define DUAM	04305
#define DUFS	04303
#define DUSM	04307
#define DUFM	04261
#define DFDH	04240
#define DFDP	04241
#define MPR	04200
#define UFM	04260
#define UFA	04300
#define UFS	04302
#define UAM	04304
#define USM	04306
#define ANA	04320
#define LAS	04340
#define SBM	04400
#define CAL	04500
#define ORA	04501
#define NZT	04520
#define LXD	04534
#define LDC	04535
#define RSCB	04540
#define RSCD	04541
#define RSCF	04542
#define RSCH	04543
#define STCB	04544
#define STCD	04545
#define STCF	04546
#define STCH	04547
#define STQ	04600
#define ORS	04602
#define DST	04603
#define SLQ	04620
#define STL	04625
#define SCD	04636
#define SXD	04634
#define PDX	04734
#define PDC	04737
#define SCHB	04640
#define SCHD	04641
#define SCHF	04642
#define SCHH	04643
#define SCDB	04644
#define SCDD	04645
#define SCDF	04646
#define SCDH	04647
#define CAD	04700	/* 704 only */
#define PXD	04754
#define PCD	04756
#define LGL	04763
#define BSF	04764
#define LGR	04765
#define RUN	04772
#define RQL	04773
#define AXC	04774
#define TRS     04775

/* Positive 0760 opcodes */
#define CLM	000000
#define LBT	000001
#define CHS	000002
#define SSP	000003
#define ENK	000004
#define IOT	000005
#define COM	000006
#define ETM	000007
#define RND	000010
#define FRN	000011
#define DCT	000012
#define RCT	000014
#define LMTM	000016
#define RDCA	001352
#define RDCB	002352
#define RDCC	003352
#define RDCD	004352
#define RDCE	005352
#define RDCF	006352
#define RDCG	007352
#define RDCH	010352
#define SLF	000140
#define SLN1	000141
#define SLN2	000142
#define SLN3	000143
#define SLN4	000144
#define SWT1	000161
#define SWT2	000162
#define SWT3	000163
#define SWT4	000164
#define SWT5	000165
#define SWT6	000166
#define BTTA	001000
#define BTTB	002000
#define BTTC	003000
#define BTTD	004000
#define BTTE	005000
#define BTTF	006000
#define BTTG	007000
#define BTTH	010000
#define PSE	0

/* Negative 0760 opcodes */
#define ETT	000000
#define PBT	000001
#define EFTM	000002
#define SSM	000003
#define LFTM	000004
#define ESTM	000005
#define ECTM	000006
#define LTM	000007
#define LSNM	000010
#define EMTM	000016
#define SLT1    000141
#define SLT2    000142
#define SLT3    000143
#define SLT4    000144
#define SLT5    000145
#define SLT6    000146
#define SLT7    000147
#define SLT8    000150

#define MSE	0

/* Opcodes */
t_opcode base_ops[] =  {
	{ TXI, "TXI", TYPE_A},
	{ TIX, "TIX", TYPE_A},
	{ TXH, "TXH", TYPE_A},
	{ STR, "STR", TYPE_E},
	{ TNX, "TNX", TYPE_A},
	{ TXL, "TXL", TYPE_A},
	{ 0, NULL, TYPE_X}
};

/* Positive opcodes */
t_opcode pos_ops[] =  {
	{ 0760, "", TYPE_P},
	{ HTR, "HTR", TYPE_B},
	{ TRA, "TRA", TYPE_B},
	{ TTR, "TTR", TYPE_B},
	{ RICA, "RICA", TYPE_B},
	{ RICC, "RICC", TYPE_B},
	{ RICE, "RICE", TYPE_B},
	{ RICG, "RICG", TYPE_B},
	{ TEFA, "TEFA", TYPE_B},
	{ TEFC, "TEFC", TYPE_B},
	{ TEFE, "TEFE", TYPE_B},
	{ TEFG, "TEFG", TYPE_B},
	{ TLQ, "TLQ", TYPE_B},
	{ IIA, "IIA", TYPE_E},
	{ TIO, "TIO", TYPE_B},
	{ OAI, "OAI", TYPE_E},
	{ PAI, "PAI", TYPE_E},
	{ TIF, "TIF", TYPE_B},
	{ IIR, "IIR", TYPE_G},
	{ RFT, "RFT", TYPE_G},
	{ SIR, "SIR", TYPE_G},
	{ RNT, "RNT", TYPE_G},
	{ RIR, "RIR", TYPE_G},
	{ TCOA, "TCOA", TYPE_B},
	{ TCOB, "TCOB", TYPE_B},
	{ TCOC, "TCOC", TYPE_B},
	{ TCOD, "TCOD", TYPE_B},
	{ TCOE, "TCOE", TYPE_B},
	{ TCOF, "TCOF", TYPE_B},
	{ TCOG, "TCOG", TYPE_B},
	{ TCOH, "TCOH", TYPE_B},
	{ TSX, "TSX", TYPE_B},
	{ TZE, "TZE", TYPE_B},
	{ CVR, "CVR", TYPE_C},
	{ TPL, "TPL", TYPE_B},
	{ XCA, "XCA", TYPE_E},
	{ TOV, "TOV", TYPE_B},
	{ TQP, "TQP", TYPE_B},
	{ TQO, "TQO", TYPE_B},
	{ MPY, "MPY", TYPE_B},
	{ VLM, "VLM", TYPE_C},
	{ DVH, "DVH", TYPE_B},
	{ DVP, "DVP", TYPE_B},
	{ VDH, "VDH", TYPE_C},
	{ VDP, "VDP", TYPE_C},
	{ FDH, "FDH", TYPE_B},
	{ FDP, "FDP", TYPE_B},
	{ FMP, "FMP", TYPE_B},
	{ DFMP, "DFMP", TYPE_B},
	{ FAD, "FAD", TYPE_B},
	{ DFAD, "DFAD", TYPE_B},
	{ FSB, "FSB", TYPE_B},
	{ DFSB, "DFSB", TYPE_B},
	{ FAM, "FAM", TYPE_B},
	{ DFAM, "DFAM", TYPE_B},
	{ FSM, "FSM", TYPE_B},
	{ DFSM, "DFSM", TYPE_B},
	{ ANS, "ANS", TYPE_B},
	{ ERA, "ERA", TYPE_B},
	{ CAS, "CAS", TYPE_B},
	{ ACL, "ACL", TYPE_B},
	{ HPR, "HPR", TYPE_E},
	{ OSI, "OSI", TYPE_B},
	{ ADD, "ADD", TYPE_B},
	{ ADM, "ADM", TYPE_B},
	{ SUB, "SUB", TYPE_B},
	{ IIS, "IIS", TYPE_B},
	{ LDI, "LDI", TYPE_B},
	{ DLD, "DLD", TYPE_B},
	{ ONT, "ONT", TYPE_B},
	{ RIS, "RIS", TYPE_B},
	{ OFT, "OFT", TYPE_B},
	{ CLA, "CLA", TYPE_B},
	{ CLS, "CLS", TYPE_B},
	{ ZET, "ZET", TYPE_B},
	{ XEC, "XEC", TYPE_B},
	{ LXA, "LXA", TYPE_B},
	{ LAC, "LAC", TYPE_B},
	{ RSCA, "RSCA", TYPE_B},
	{ RSCC, "RSCC", TYPE_B},
	{ RSCE, "RSCE", TYPE_B},
	{ RSCG, "RSCG", TYPE_B},
	{ STCA, "STCA", TYPE_B},
	{ STCC, "STCC", TYPE_B},
	{ STCE, "STCE", TYPE_B},
	{ STCG, "STCG", TYPE_B},
	{ LDQ, "LDQ", TYPE_B},
	{ ENB, "ENB", TYPE_B},
	{ STZ, "STZ", TYPE_B},
	{ STO, "STO", TYPE_B},
	{ SLW, "SLW", TYPE_B},
	{ STI, "STI", TYPE_B},
	{ STA, "STA", TYPE_B},
	{ STD, "STD", TYPE_B},
	{ STT, "STT", TYPE_B},
	{ STP, "STP", TYPE_B},
	{ SXA, "SXA", TYPE_B},
	{ SCA, "SCA", TYPE_B},
	{ SCHA, "SCHA", TYPE_B},
	{ SCHC, "SCHC", TYPE_B},
	{ SCHE, "SCHE", TYPE_B},
	{ SCHG, "SCHG", TYPE_B},
	{ SCDA, "SCDA", TYPE_B},
	{ SCDC, "SCDC", TYPE_B},
	{ SCDE, "SCDE", TYPE_B},
	{ SCDG, "SCDG", TYPE_B},
	{ PAX, "PAX", TYPE_F},
	{ PAC, "PAC", TYPE_F},
	{ PXA, "PXA", TYPE_F},
	{ PCA, "PCA", TYPE_F},
	{ NOP, "NOP", TYPE_E},
	{ RDS, "RDS", TYPE_I},
	{ BSR, "BSR", TYPE_I},
	{ LLS, "LLS", TYPE_D},
	{ LRS, "LRS", TYPE_D},
	{ WRS, "WRS", TYPE_I},
	{ ALS, "ALS", TYPE_D},
	{ WEF, "WEF", TYPE_I},
	{ ARS, "ARS", TYPE_D},
	{ REW, "REW", TYPE_I},
	{ AXT, "AXT", TYPE_D},
        { DSR, "DSR", TYPE_D},
	{ 0, NULL, TYPE_X}
};
 
/* Negative opcodes */
t_opcode neg_ops[] =  {
	{ 04760, "", TYPE_N},
	{ RICB, "RICB", TYPE_B},
	{ RICD, "RICD", TYPE_B},
	{ RICF, "RICF", TYPE_B},
	{ RICH, "RICH", TYPE_B},
	{ TEFB, "TEFB", TYPE_B},
	{ TEFD, "TEFD", TYPE_B},
	{ TEFF, "TEFF", TYPE_B},
	{ TEFH, "TEFH", TYPE_B},
	{ RIA, "RIA", TYPE_B},
	{ PIA, "PIA", TYPE_E},
	{ IIL, "IIL", TYPE_G},
	{ LFT, "LFT", TYPE_G},
	{ SIL, "SIL", TYPE_G},
	{ LNT, "LNT", TYPE_G},
	{ RIL, "RIL", TYPE_G},
	{ TCNA, "TCNA", TYPE_B},
	{ TCNB, "TCNB", TYPE_B},
	{ TCNC, "TCNC", TYPE_B},
	{ TCND, "TCND", TYPE_B},
	{ TCNE, "TCNE", TYPE_B},
	{ TCNF, "TCNF", TYPE_B},
	{ TCNG, "TCNG", TYPE_B},
	{ TCNH, "TCNH", TYPE_B},
	{ TNZ, "TNZ", TYPE_B},
	{ CAQ, "CAQ", TYPE_C},
	{ TMI, "TMI", TYPE_B},
	{ XCL, "XCL", TYPE_E},
	{ TNO, "TNO", TYPE_B},
	{ CRQ, "CRQ", TYPE_C},
	{ DUFA, "DUFA", TYPE_B},
	{ DUAM, "DUAM", TYPE_B},
	{ DUFS, "DUFS", TYPE_B},
	{ DUSM, "DUSM", TYPE_B},
	{ DUFM, "DUFM", TYPE_B},
	{ DFDH, "DFDH", TYPE_B},
	{ DFDP, "DFDP", TYPE_B},
	{ MPR, "MPR", TYPE_B},
	{ UFM, "UFM", TYPE_B},
	{ UFA, "UFA", TYPE_B},
	{ UFS, "UFS", TYPE_B},
	{ UAM, "UAM", TYPE_B},
	{ USM, "USM", TYPE_B},
	{ ANA, "ANA", TYPE_B},
	{ LAS, "LAS", TYPE_B},
	{ SBM, "SBM", TYPE_B},
	{ CAL, "CAL", TYPE_B},
	{ ORA, "ORA", TYPE_B},
	{ NZT, "NZT", TYPE_B},
	{ LXD, "LXD", TYPE_B},
	{ LDC, "LDC", TYPE_B},
	{ RSCB, "RSCB", TYPE_B},
	{ RSCD, "RSCD", TYPE_B},
	{ RSCF, "RSCF", TYPE_B},
	{ RSCH, "RSCH", TYPE_B},
	{ STCB, "STCB", TYPE_B},
	{ STCD, "STCD", TYPE_B},
	{ STCF, "STCF", TYPE_B},
	{ STCH, "STCH", TYPE_B},
	{ STQ, "STQ", TYPE_B},
	{ ORS, "ORS", TYPE_B},
	{ DST, "DST", TYPE_B},
	{ SLQ, "SLQ", TYPE_B},
	{ STL, "STL", TYPE_B},
	{ SCD, "SCD", TYPE_B},
	{ SXD, "SXD", TYPE_B},
	{ PDX, "PDX", TYPE_F},
	{ PDC, "PDC", TYPE_F},
	{ SCHB, "SCHB", TYPE_B},
	{ SCHD, "SCHD", TYPE_B},
	{ SCHF, "SCHF", TYPE_B},
	{ SCHH, "SCHH", TYPE_B},
	{ SCDB, "SCDB", TYPE_B},
	{ SCDD, "SCDD", TYPE_B},
	{ SCDF, "SCDF", TYPE_B},
	{ SCDH, "SCDH", TYPE_B},
	{ PXD, "PXD", TYPE_F},
	{ PCD, "PCD", TYPE_F},
	{ LGL, "LGL", TYPE_D},
	{ BSF, "BSF", TYPE_I},
	{ LGR, "LGR", TYPE_D},
	{ RUN, "RUN", TYPE_I},
	{ RQL, "RQL", TYPE_D},
	{ AXC, "AXC", TYPE_D},
        { TRS, "TRS", TYPE_D},
	{ 0, NULL, TYPE_X}
};

/* Positive 0760 opcodes */
t_opcode pos_760[] =  {
	{ CLM, "CLM", TYPE_E},
	{ ETT, "BTT", TYPE_D},
	{ LBT, "LBT", TYPE_E},
	{ CHS, "CHS", TYPE_E},
	{ SSP, "SSP", TYPE_E},
	{ ENK, "ENK", TYPE_E},
	{ IOT, "IOT", TYPE_E},
	{ COM, "COM", TYPE_E},
	{ ETM, "ETM", TYPE_E},
	{ RND, "RND", TYPE_E},
	{ FRN, "FRN", TYPE_E},
	{ DCT, "DCT", TYPE_E},
	{ RCT, "RCT", TYPE_E},
	{ LMTM, "LMTM", TYPE_E},
	{ RDCA, "RDCA", TYPE_E},
	{ RDCB, "RDCB", TYPE_E},
	{ RDCC, "RDCC", TYPE_E},
	{ RDCD, "RDCD", TYPE_E},
	{ RDCE, "RDCE", TYPE_E},
	{ RDCF, "RDCF", TYPE_E},
	{ RDCG, "RDCG", TYPE_E},
	{ RDCH, "RDCH", TYPE_E},
	{ SLF, "SLF", TYPE_E},
	{ SLN1, "SLN 1", TYPE_E},
	{ SLN2, "SLN 2", TYPE_E},
	{ SLN3, "SLN 3", TYPE_E},
	{ SLN4, "SLN 4", TYPE_E},
	{ SWT1, "SWT 1", TYPE_E},
	{ SWT2, "SWT 2", TYPE_E},
	{ SWT3, "SWT 3", TYPE_E},
	{ SWT4, "SWT 4", TYPE_E},
	{ SWT5, "SWT 5", TYPE_E},
	{ SWT6, "SWT 6", TYPE_E},
	{ BTTA, "BTTA", TYPE_E},
	{ BTTB, "BTTB", TYPE_E},
	{ BTTC, "BTTC", TYPE_E},
	{ BTTD, "BTTD", TYPE_E},
	{ BTTE, "BTTE", TYPE_E},
	{ BTTF, "BTTF", TYPE_E},
	{ BTTG, "BTTG", TYPE_E},
	{ BTTH, "BTTH", TYPE_E},
	{ PSE, "PSE", TYPE_D},
	{ 0, NULL, TYPE_X}
};

/* Negative 0760 opcodes */
t_opcode neg_760[] =  {
	{ ETT, "ETT", TYPE_D},
	{ PBT, "PBT", TYPE_D},
	{ EFTM, "EFTM", TYPE_E},
	{ SSM, "SSM", TYPE_E},
	{ LFTM, "LFTM", TYPE_E},
	{ ESTM, "ESTM", TYPE_E},
	{ ECTM, "ECTM", TYPE_E},
	{ LTM, "LTM", TYPE_E},
	{ EMTM, "EMTM", TYPE_E},
	{ SLT1, "SLT 1", TYPE_E},
	{ SLT2, "SLT 2", TYPE_E},
	{ SLT3, "SLT 3", TYPE_E},
	{ SLT4, "SLT 4", TYPE_E},
	{ SLT5, "SLT 5", TYPE_E},
	{ SLT6, "SLT 6", TYPE_E},
	{ SLT7, "SLT 7", TYPE_E},
	{ MSE, "MSE", TYPE_D},
	{ 0, NULL, TYPE_X}
};

t_sym ibsys_syms[] = {
	00100, "SYSTRA",
	00101, "SYSDAT",
	00102, "SYSCUR",
	00103, "SYSRET",
	00104, "SYSKEY",
	00105, "SYSSWS",
	00106, "SYSPOS",
	00107, "SYSUNI",
	00110, "SYSUBC",
	00111, "SYSUAV",
	00112, "SYSUCW",
	00113, "SYSRPT",
	00114, "SYSCEM",
	00115, "SYSDMP",
	00116, "SYSIOX",
	00117, "SYSIDR",
	00120, "SYSCOR",
	00121, "SYSLDR",
	00122, "SYSACC",
	00123, "SYSPID",
	00124, "SYSCYD",
	00126, "SYSSLD",
	00127, "SYSTCH",
	00131, "SYSTWT",
	00132, "SYSGET",
	00133, "SYSJOB",
	00134, ".CHEXI",
	00135, ".MODSW",
	00140, "SYSLB1",
	00141, "SYSLB2",
	00142, "SYSLB3",
	00143, "SYSLB4",
	00144, "SYSCRD",
	00145, "SYSPRT",
	00146, "SYSPCH",
	00147, "SYSOU1",
	00150, "SYSOU2",
	00151, "SYSIN1",
	00152, "SYSIN2",
	00153, "SYSPP1",
	00154, "SYSPP2",
	00155, "SYSCK1",
	00156, "SYSCK2",
	00157, "SYSUT1",
	00160, "SYSUT2",
	00161, "SYSUT3",
	00162, "SYSUT4",
	00702, "(ACTIV",
	00703, "(ACTVX",
	00704, "(NDATA",
	00705, "(NDSLX",
	00706, "(PROUT",
	00707, "(PUNCH",
	00710, "(ENBSW",
	00711, "(PAWSX",
	00712, "(PAUSE",
	00713, "(STOPX",
	00714, "(SYMUN",
	00715, "(DECVD",
	00716, "(DECVA",
	00717, "(CKWAT",
	00720, "(BCD5R",
	00721, "(BCD5X",
	00722, "(CVPRT",
	00723, "(STOPD",
	00724, "(CHXAC",
	00725, "(URRXI",
	00726, "(RCTXI",
	00727, "(RCHXI",
	00730, "(TCOXI",
	00731, "(TRCXI",
	00732, "(ETTXI",
	00733, "(TEFXI",
	00734, "(TRAPX",
	00735, "(TRAPS",
	00736, "(COMMM",
	00737, "(LTPOS",
	00740, "(IOXSI",
	00741, "(CHPSW",
	00742, "(TRPSW",
	00743, "(FDAMT",
	00744, "(SDCXI",
	00745, "(STCXI",
	00746, "(COMMD",
	00747, "(IBCDZ",
	00750, "(CHXSP",
	03720, "SYSORG",
	0, 0,
};

void print_sym(FILE *of, long val, int rel) 
{
	
	if (rel == 2 && (ibsys | absimg)) {
	   int	i;
	   if (val < 0100) {
		fprintf(of, "%d", val);
		return;
	   }
	   for (i = 0; ibsys_syms[i].name != 0; i++) {
		if (ibsys_syms[i].value == val) {
		    fputs(ibsys_syms[i].name, of);
		    return;
		}
	   }   
	}
	switch (rel) {
	case 0:
		if (val & 040000) 
			fprintf (of, "-%d", 0100000 - val);
	        else
		        fprintf (of, "%d", val);
		break;
	case 2:
		fprintf (of, "L%o", val);
		break;
	case 3:
		fprintf (of, "R%o", val);
		break;
	}
}
	

static const char *chname[] = {
 "A", "B", "C", "D", "E", "F", "G", "H", NULL };

void lookup_sopcode(FILE *of, long long val, t_opcode *tab) 
{
    int		op = val & 0777777;
    while(tab->name != NULL) {
	if (tab->opbase == op) {
	   fputs(tab->name, of);
	   switch(tab->type) {
	   case TYPE_D: fprintf (of, " %d", val & 0000000077777L); 
			if ((val & 0000000700000L) != 0) {
    			    fputc(',', of);
    			    fputc('0' + (7 & (val >> 15)), of);
			}
			return;
	   case TYPE_E:
			if ((val & 0000000700000L) != 0) {
    			    fputc(' ', of);
    			    fputc('0' + (7 & (val >> 15)), of);
			}
			return;
	   default:
			fputs("       ", of);
			return;
	   }
	}
	tab++;
   }
   tab--;
   fprintf (of, "%s %05o", tab->name, val & 0000000077777L); 
   if ((val & 0000000700000L) != 0) {
        fputc(',', of);
        fputc('0' + (7 & (val >> 15)), of);
   }
}


	

void lookup_opcode(FILE *of, long long val, t_opcode *tab, int rela, int relb) 
{
    int		op = (val >> 24) & 07777;

    if (val == 0)
	return;
    if (op == HTR && (val & 007777777777L) != 0) {
	fputs("PZE    ", of);
   	print_sym (of, val & 0000000077777L, rela); 
	fputc(',', of);
	if ((val & 0000000700000L) != 0) 
    	    fputc('0' + (7 & (val >> 15)), of);
	fputc(',', of);
        print_sym (of, (val >> 18) & 0000000077777L, relb); 
	return;
    }
	
    while(tab->name != NULL) {
	if (tab->opbase == op) {
	   fputs(tab->name, of);
	   switch(tab->type) {
	   case TYPE_B: if ((val & 000060000000L) == 000060000000L) 
			   fputc('*', of);
			  fputs("   ", of);
   			print_sym(of, val & 0000000077777L, rela); 
			if ((val & 0000000700000L) != 0) {
    			    fputc(',', of);
    			    fputc('0' + (7 & (val >> 15)), of);
			}
			return;
	   case TYPE_C: 
			fputs("   ", of);
			print_sym(of, val & 0000000077777L, rela);
			fprintf (of, ", %d", (val >> 17) & 0000777L);
			if ((val & 0000000700000L) != 0) {
    			    fputc(',', of);
    			    fputc('0' + (7 & (val >> 15)), of);
			}
			return;
	   case TYPE_E:
			if ((val & 0777777L) == 0) {
			      fputs("        ",of);
			      return;
			}
	   case TYPE_D: fputs("   ", of);
			print_sym (of, val & 0000000077777L, (ibsys|absimg)? 0: rela); 
			if ((val & 0000000700000L) != 0) {
    			    fputc(',', of);
    			    fputc('0' + (7 & (val >> 15)), of);
			}
			return;
	   case TYPE_F: fputs("   ", of);
			if (rela != 0 || (val & 077777L) != 0)
				print_sym(of, val & 077777L, (ibsys|absimg)? 0: rela);
    			fputc(',', of);
    			fputc('0' + (7 & (val >> 15)), of);
			return;
	   case TYPE_G: fputs("   ", of);
			fprintf (of, "%06o   ", val & 0000000777777L); 
			if (rela != 0 && !(ibsys|absimg))
			    fputc('*', of);
			return;
	   case TYPE_P: lookup_sopcode(of, val, pos_760);
		        return;
	   case TYPE_N: lookup_sopcode(of, val, neg_760);
		        return;
	   case TYPE_I:	switch (val & 0760) {
			case 0200: fputc('D', of); break;
			case 0220: fputc('B', of); break;
			case 0340: fputc('U', of); break;
			case 0360: fputc('P', of); break;
			case 0320: fputc('C', of); break;
			default:	fprintf(of, "%o", val & 0760); break;
			}
			fputc('A' + ((val >> 9) & 7) - 1, of);
			fprintf(of, "   %d", val & 017);
			return;
 	   default:    return;
	   }
	}
	tab++;
   }
   fputs("PZE   ", of);
   print_sym (of, val & 0000000077777L, rela); 
   if ((val & 0777777700000L) != 0) {
       fputc(',', of);
       if ((val & 0000000700000L) != 0) {
           fputc('0' + (7 & (val >> 15)), of);
       }
       if ((val & 0777777000000L) != 0)  {
           fputc(',', of);
           print_sym (of, (val >> 18) & 0000000077777L, relb); 
       }
   }
}


/* Symbolic decode

   Inputs:
	*of	=	output stream
	addr	=	current PC
	*val	=	pointer to values
	*uptr	=	pointer to unit
	sw	=	switches
   Outputs:
	return	=	status code
*/

void fprint_sym (FILE *of, long long inst, int rela, int relb) {

    switch(07&(inst >> 33)) {
    case TXI: fputs("TXI    ", of); goto type_a;
    case TIX: fputs("TIX    ", of); goto type_a;
    case TXH: fputs("TXH    ", of); goto type_a;
    case STR: fputs("STR    ", of); goto type_a;
    case TNX: fputs("TNX    ", of); goto type_a;
    case TXL: fputs("TXL    ", of); goto type_a;

type_a:
    print_sym (of, inst & 0000000077777L, rela); 
    fputc(',', of);
    if (((inst >> 15) & 7) != 0)
        fputc('0' + (7 & (inst >> 15)), of);
    fputc(',', of);
    print_sym (of, (inst >> 18) & 0000000077777L, relb); 
    break;
    case 04: lookup_opcode(of, inst, neg_ops, rela, relb); break;
    case 00: lookup_opcode(of, inst, pos_ops, rela, relb); break;
    }
/* Print value in octal first */
fputs("   OCT ", of);
if (inst & 0400000000000L)
    fputc('-', of);
else
    fputc(' ', of);
fprintf(of, "%012llo\t", inst & 0377777777777L); 

}
/***********************************************************************
* octdump - Octal dump routine.
***********************************************************************/

int	externals = 0;
int	progcard = 0;
static void
octdump(FILE *file, void *ptr, int size,int offset)
{
   int jjj;
   int iii;
   int dmpmax;
   int addr;
   int len;
   int rm;
   long long	wd;
   long long	relo;
   unsigned char *tp;
   unsigned char *cp;
   unsigned char *rp;
   unsigned char *xp;

   tp = (unsigned char *)ptr;
   cp = (unsigned char *)ptr;

   if (ibsys) {
        size /= 6;
        while(size > 0) {
           for (wd = 0, jjj = 0; jjj < 6; jjj++) {
               wd <<= 6;
               wd |= *cp++ & 077;
           }
           size--;
           tp += 6;
           addr = wd & 077777;
           len = (wd >> 18) & 077777;
           iii = 0;
           if (len == 0) {
                fprintf(file, "\tEND\tL%05o\n", addr);
           } else {
                fprintf(file, "\tORG\t%d\n", addr);
                for (; iii < (len) && size >= 0; iii++) {
                  size--;
                  fprintf (file, "L%-5o   ", addr++);
    
                  for (wd = 0, jjj = 0; jjj < 6; jjj++) {
                     wd <<= 6;
                     wd |= *cp++ & 077;
                  }
                  fprint_sym (file, wd, 2, 0);
                  fprintf ((file), "\t\tBCD 1");
                  for (jjj = 0; jjj < 6; jjj++) {
                     fprintf (file, "%c", ibsys_ascii[*tp++ & 077]);
                  }
                  fprintf ((file), "\n");
               }
           }
        }
   } else {
       size /= 6;
       for (wd = 0, jjj = 0; jjj < 6; jjj++) {
    	wd <<= 6;
    	wd |= *cp++ & 077;
       }
	tp += 6;
       iii = (wd >> 33) & 07;
       jjj = (wd >> 15) & 07;
       addr = wd & 077777;
       len = (wd >> 18) & 077777;
       fprintf ((file), "%o %05o %o %05o\n", iii, len, jjj, addr);
	len &= 037;
	switch (iii) {
	case 0:
		progcard = 0;
              if (len == 0) {
    	       addr = 0;
    	       iii = 0;
    	       cp = (unsigned char *)ptr;
              } else if ((len + 3) == size) {
    	       fprintf (file, "Wrong len %08o vs %08o\n", size, len);
    	       addr -= size - len;
    	       iii = 0;
                   cp = (unsigned char *)ptr;
              } else if (!image) {
    	       cp+=6;
    	       tp+=6;
                 fprintf ((file), "abs\tOCT %012llo\tBCD 1", wd);
                 for (jjj = 0; jjj < 6; jjj++) {
    	         fprintf (file, "%c", ibsys_ascii[*tp++ & 077]);
                 }
                 fprintf ((file), "\n");
              }
              for (; iii < (size); iii++) {
                 fprintf (file, "L%05o   ", addr++);
           
                 for (wd = 0, jjj = 0; jjj < 6; jjj++) {
    	       wd <<= 6;
    	       wd |= *cp++ & 077;
                 }
                 fprint_sym (file, wd, 2, 2);
                 fprintf ((file), "\t\tBCD 1");
                 for (jjj = 0; jjj < 6; jjj++) {
    	         fprintf (file, "%c", ibsys_ascii[*tp++ & 077]);
                 }
                 fprintf ((file), "\n");
       	      }
	      break;
	case 2: 
		progcard = 0;
              if (len == 0) {
    	       addr = 0;
    	       iii = 0;
    	       cp = (unsigned char *)ptr;
              } else if ((len + 3) == size) {
    	       fprintf (file, "Wrong len %08o vs %08o\n", size, len);
    	       addr -= size - len;
    	       iii = 0;
                   cp = (unsigned char *)ptr;
              } else if (!image) {
    	       cp+=6;
    	       tp+=6;
		iii = 6;
              }
	        rp = cp;
		rm = 040;
       //          for (relo = 0, jjj = 0; jjj < 10; jjj++) {
    	//       relo <<= 6;
    	 //      relo |= *cp++ & 077;
          //       }
		cp += 12;
		tp += 12;
		len += 6;
//                 fprintf ((file), "rel\t %012llo\n", relo);
              for (; iii < len; iii++) {
		 int rela, relb;
           
                 for (wd = 0, jjj = 0; jjj < 6; jjj++) {
    	             wd <<= 6;
    	             wd |= *cp++ & 077;
                 }
		 if (externals == 0) {
		   if (absimg) {
                     rela = 2;
		     relb = 0;
		   } else {
		     rela = relb = 0;
		     if (*rp & rm) {
		     	rm >>= 1;
			if (rm == 0) {
			    rm = 040;
			    rp++;
			}
			relb = 2 | ((*rp & rm) ? 1:0);
		     }
		     rm >>= 1;
		     if (rm == 0) {
		        rm = 040;
		        rp++;
		     }
		     if (*rp & rm) {
		     	rm >>= 1;
			if (rm == 0) {
			    rm = 040;
			    rp++;
			}
			rela = 2 | ((*rp & rm) ? 1:0);
		     }
		     rm >>= 1;
		     if (rm == 0) {
		        rm = 040;
		        rp++;
		     }
                    /* rela = (020000000000000000000LL & relo) != 0;
		     relo <<= 1;
		     if (rela) {
                         rela = 2 | ((020000000000000000000LL & relo) != 0);
		         relo <<= 1;
		     }
                     relb = (020000000000000000000LL & relo) != 0;
		     relo <<= 1;
		     if (relb) {
                         relb = 2 | ((020000000000000000000LL & relo) != 0);
		         relo <<= 1;
		     } */
		   }
		   if (rela == 0) {
		   	if (relb == 0) 
                           fprintf (file, "K%-5o ", addr++);
			else
                           fprintf (file, "R%-5o ", addr++);
	           } else {
		       if (relb == 0) 
                           fprintf (file, "L%-5o ", addr++);
		       else
                           fprintf (file, "B%-5o ", addr++);
	           }
//		     fprintf ((file), "r%o %o ", rela, relb);
	
                     fprint_sym (file, wd, rela, relb);
//		     if ((rela | relb) == 0) {
                         fprintf ((file), "\t\tBCD 1");
                         for (jjj = 0; jjj < 6; jjj++) {
    	                     fprintf (file, "%c", ibsys_ascii[*tp++ & 077]);
		         }
//		      } else {
//			 tp += 6;
//		      }
		 } else {
		     relo <<= 1;
		     relo <<= 1;
		     fprintf ((file), "=%-5o ", addr);
		     xp = tp;
                     for (jjj = 0; jjj < 6; jjj++) {
    	                 fprintf (file, "%c", ibsys_ascii[*xp++ & 077]);
                     }
                     fprintf ((file), "\n");
		     fprintf ((file), "L%-5o  EXTERN     ", addr++);
                     for (jjj = 0; jjj < 6; jjj++) {
    	                 fprintf (file, "%c", ibsys_ascii[*tp++ & 077]);
                     }
		     externals--;
		 }
                 fprintf ((file), "\n");
       	      }
	      break;
	case 4:
    	      iii = 0;
              if (len == 0) {
    	       addr = 0;
    	       cp = (unsigned char *)ptr;
              } else {
    	       cp+=6;	/* Skip checksum */
    	       tp+=6;
              }
		if (progcard == 0) {
                    for (wd = 0, jjj = 0; jjj < 6; jjj++) {
    	                wd <<= 6;
    	                wd |= *cp++ & 077;
                    }
                    fprintf ((file), "prog %012llo\n", wd);
		    externals = (wd >> 18) & 077777;
		tp += 12;
		cp += 6;
		iii += 2;
		progcard = 1;
	      }
              for (; iii < (len); iii++) {
                 for (wd = 0, jjj = 0; jjj < 6; jjj++) {
    	       wd <<= 6;
    	       wd |= *cp++ & 077;
                 }
		 if ((iii & 1)) {
		   xp = tp;
		 fprintf ((file), "=%-5o ", wd & 077777);
                 for (jjj = 0; jjj < 6; jjj++) {
    	         fprintf (file, "%c", ibsys_ascii[*xp++ & 077]);
		 }
                 fprintf ((file), "\n  L%05o", wd & 077777);
                 fprintf ((file), "  ENTRY    ");
                 for (jjj = 0; jjj < 6; jjj++) {
    	         fprintf (file, "%c", ibsys_ascii[*tp++ & 077]);
		}
                 fprintf ((file), "\n");
		tp+= 6;
		}
       	      }
	      break;
        }
    }

}


/***********************************************************************
* main - Main procedure.
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *bp;
   char *file;
   int i;
   long int size;
   int org;
   int ch;
   char tape[MAXRECSIZE];

   /*
   ** Process arguments and file.
   */

   raw = 0;
   file = NULL;
   recmode = 0;
   altchars = 0;
   recnum = 1;
   filenum = 1;
   tapformat = 0;
   image = 0;
   ibsys = 0;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];
      if (*bp == '-')
      {
	 for (bp++; *bp; bp++) switch (*bp)
	 {
	 case 'c':
	    colbin = 1;
	    break;
         case 'b':
            absimg = 1;
            break;
	 case 'a':
	    altchars = 1;
	    break;
	 case 'i':
	    image = 1;
	    break;
         case 'r':
	    raw = 1;
	    break;
         case 't':
	    tapformat = 1;
	    break;
	 case 's':
	    ibsys = 1;
	    break;
	 default:
      usage:
	    fprintf (stderr, "usage: bd [options] file\n");
	    fprintf (stderr, " options:\n");
	    fprintf (stderr, "    -a           - Use Alt charset \n");
	    exit (1);
	 }
      }
      else
      {
	 if (file) goto usage;
	 file = bp;
      }
   }

   if (!file) goto usage;

   /*
   ** Open input file.
   */

   if ((fd = fopen (file, "rb")) == NULL) {
      perror ("Can't open");
      exit (1);
   }

   /*
   ** Dump the file.
   */

   size = 0;
   org = 0;
   printf ("Dump of %s:\n", file);
   memset (tape, '\0', sizeof(tape));
   if (tapformat) {
	while(fread(&size, sizeof(long int), 1, fd) != 0) {
		if (size == 0) {
		    filenum++;
		    puts("*EOF*\n");
		    recnum = 1;
		} else {
		    if (fread(tape, 1, size, fd) != size)  {
		      fprintf(stderr, "read error\n");
		      break;
		    }
	            fread(&size, sizeof(long int), 1, fd);
	            printf ("File %d record %d: (%d)\n", filenum, recnum++, size);
		    if (colbin && size == 168) {
		       unsigned char record[144];
		       int	      i, j, k;
		       memset(&record, 0, sizeof(record));
                       /* Bit flip into read buffer */
                       for(i = 0; i < 72; i++) {
			  int bit = 6 - ((i/2)%6);
			  for(j = 0; j < 6; j++) {
			     if (tape[(i*2)+1 ] & (1<<j))
				record[i+(j*12)] |= bit;
			  }
			  for(j = 0; j < 6; j++) {
			     if (tape[(i*2)] & (1<<j))
				record[i + 24 + (j*12)] |= bit;
			  }
                       }
		       memcpy(tape, &record, sizeof(record));
		       size = sizeof(record);
                    }
	            octdump (stdout, tape, size, org);
	            memset (tape, '\0', sizeof(tape));
	            org = 0;
	            size = 0;
	       }
	}
   } else if (raw) {
	while((ch = fgetc (fd)) != EOF) {
	    if (colbin && size == 144) {
#if 0
	       unsigned char record[144];
	       int	      i, j, k;
      printf ("File %d record %d:\n", filenum, recnum++);
	       memset(&record, 0, sizeof(record));
               /* Bit flip into read buffer */
               for(i = 0; i < 24; i++) {
                  int bit = 1 << (i/2);
                  long long mask = 1;
                  long long wd = 0;
                  int b = 2 * 36 * (i & 1);
		  int o = 1;
		  int col;
		  if (bit & 07700) {
		     o = 0;
		     bit >>= 6;
		  }
                  for(col = 35; col >= 0; col--, mask <<= 1) {
                     if (tape[o + 2*col + b] & bit)
                         wd |= mask;
                  }
	          for(col = 5; col >= 0; col--) 
		     record[6*i + col] = 077 & (wd >> (6 * (5 - col)));
               }

	       memcpy(tape, &record, sizeof(record));
	       size = sizeof(record);
#endif
	       octdump (stdout, tape, size, org);
	       memset (tape, '\0', size);
	       while(size < 160 && fgetc (fd) != EOF) size++;
	       org = 0;
	       size = 0;
	    }
	    tape[size++] = ch & 077;
	}
   } else {
       while ((ch = fgetc (fd)) != EOF) {
         if ((ch & 0200) || (!altchars && ch == 072)) {
	    if (size) {
	       printf ("File %d record %d:\n", filenum, recnum++);
	       octdump (stdout, tape, size, org);
	       memset (tape, '\0', sizeof(tape));
	       org = 0;
	       size = 0;
	       if (ch == 0217) {
		  printf ("File %d EOF\n", filenum++);
		  recnum = 1;
	       } else
		  tape[size++] = ch & 0xFF;
	    } else 
	       tape[size++] = ch & 0xFF;
	 } else
	    tape[size++] = ch & 0xFF;
      }
   }
   if (size) {
      printf ("File %d record %d:\n", filenum, recnum++);
      octdump (stdout, tape, size, org);
   }


   fclose (fd);
   return (0);
   
}

