/* i7090_cdp.c: IBM 7090 Card punch.

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   This is the standard card punch.

*/

#include "i7090_defs.h"

#define NUM_DEVS_CDP	4
#define UNIT_CDP	UNIT_ATTABLE | UNIT_DISABLE

/* Flags for punch and reader. */
#define UNIT_V_MODE	(UNIT_V_LOCAL)
#define UNIT_MODE	(3 << UNIT_V_MODE)
#define MODE_AUTO	(0 << UNIT_V_MODE)
#define MODE_BIN	(1 << UNIT_V_MODE)
#define MODE_TEXT	(2 << UNIT_V_MODE)

/* std devices. data structures

   chan_dev	Channel device descriptor
   chan_unit	Channel unit descriptor
   chan_reg	Channel register list
   chan_mod	Channel modifiers list
*/

extern uint8        iocheck;

#define CDPSTA_READ	000001	/* Unit is in read */
#define CDPSTA_WRITE	000002	/* Unit is in write */
#define CDPSTA_ON	000004	/* Unit is running */
#define CDPSTA_EOF	000010	/* Hit end of file */
#define CDPSTA_IDLE	000040	/* Unit between operation */
#define CDPSTA_CMD	000100	/* Unit has recieved a cmd */
#define CDPSTA_PUNCH	000200	/* Punch strobe during run */
#define CDPSTA_POSMASK  077000
#define CDPSTA_POSSHIFT	9

struct _cdp_data
{
    t_uint64            wbuff[24];	/* Card buffer */
}
cdp_data[NUM_DEVS_CDP];

uint32              cdp(UNIT *, uint16, uint16);
t_stat              cdp_srv(UNIT *);
void                cdp_ini(UNIT *, t_bool);
t_stat              cdp_reset(DEVICE *);
t_stat              cdp_attach(UNIT *, char *);
t_stat              cdp_detach(UNIT *);
extern t_stat       chan_boot(int32, DEVICE *);
extern char         six_to_ascii[64];

DIB                 cdp_dib = { CHAN_7607, 1, 0341, 0777, &cdp, &cdp_ini };
UNIT                cdp_unit[] = {
    {UDATA(&cdp_srv, UNIT_S_CHAN(1) | UNIT_CDP, 0), 6000},	/* A */
    {UDATA(&cdp_srv, UNIT_S_CHAN(2) | UNIT_CDP, 0), 6000},	/* B */
    {UDATA(&cdp_srv, UNIT_S_CHAN(3) | UNIT_CDP | UNIT_DIS, 0), 6000},	/* C */
    {UDATA(&cdp_srv, UNIT_S_CHAN(0) | UNIT_CDP | UNIT_DIS, 0), 6000},	/* D */
};

MTAB                cdp_mod[] = {
    {UNIT_MODE, MODE_AUTO, "AUTO", "AUTO", NULL, NULL, NULL},
    {UNIT_MODE, MODE_BIN, "BIN", "BIN", NULL, NULL, NULL},
    {UNIT_MODE, MODE_TEXT, "TEXT", "TEXT", NULL, NULL, NULL},
    
	{MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "CHAN", "CHAN", &set_chan,
     &get_chan, NULL},
    {0}
};

DEVICE              cdp_dev = {
    "CP", cdp_unit, NULL, cdp_mod,
    NUM_DEVS_CDP, 8, 15, 1, 8, 36,
    NULL, NULL, &cdp_reset, NULL, &cdp_attach, &cdp_detach,
    &cdp_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};

/* Card punch routine

   Modifiers have been checked by the caller
   C modifier is recognized (column binary is implemented)
*/

t_stat
punch_card(UNIT * uptr, int chan, int unit)
{
/* Convert word record into column image */
/* Check output type, if auto or text, try and convert record to bcd first */
/* If failed and text report error and dump what we have */
/* Else if binary or not convertable, dump as image */

    int32               i;
    uint16              buff[80];	/* + null */

    if ((uptr->flags & UNIT_ATT) == 0)
	return SCPE_UNATT;	/* attached? */

    memset(buff, 0, sizeof(buff));
    /* Bit flip into temp buffer */
    for (i = 0; i < 24; i++) {
	int                 bit = 1 << (i / 2);
	t_uint64            mask = 1;
	t_uint64            wd = 0;
	int                 b = 36 * (i & 1);
	int                 col;

	wd = cdp_data[unit].wbuff[i];
	for (col = 35; col >= 0; mask <<= 1, col--) {
	    if (wd & mask)
		buff[col + b] |= bit;
	}
	cdp_data[unit].wbuff[i] = 0;
    }

    if ((uptr->flags & UNIT_MODE) != MODE_BIN) {
	/* Try to convert to text */
	char                out[83];
	int                 ok = 1;

	/* Scan each column */
	for (i = 0; i < 80; i++) {
	    int                 digit = buff[i] & 0x1ff;
	    int                 bcd = 0;

	    if (digit & 0x2) {
		bcd += 8;
		digit &= ~0x2;
	    }
	    while (digit != 0 && (digit & 0x200) == 0) {
		bcd++;
		digit <<= 1;
	    }
	    if ((digit & 0x1ff) != 0) {
		out[i] = 0x7f;
		ok = 0;
		continue;
	    }
	    switch (buff[i] & 0xe00) {
	    case 0x000:
		break;
	    case 0x200:
		bcd += 060;
		break;
	    case 0x400:
		bcd += 040;
		break;
	    case 0x800:
		bcd += 020;
		break;
	    default:
		out[i] = 0x7f;
		ok = 0;
		continue;
	    }
	    /* unmess the conversion code */
	    if (bcd == 0)
		bcd = 060;
	    else if (bcd == 060)
		bcd = 0;
	    out[i] = six_to_ascii[bcd];
	}
	/* Trim off trailing spaces */
	while (i > 0 && out[--i] == ' ') ;
	out[++i] = '\n';
	out[++i] = '\0';
	if ((uptr->flags & UNIT_MODE) == MODE_TEXT && !ok) {
	    iocheck = 1;
	}
	if ((uptr->flags & UNIT_MODE) == MODE_TEXT ||
	    ((uptr->flags & UNIT_MODE) == MODE_AUTO && ok)) {
	    sim_fwrite(out, 1, i, uptr->fileref);
	    uptr->u5 &= ~CDPSTA_PUNCH;
	    return SCPE_OK;
	}
    }
/* Binary data, just dump out */
    uptr->u5 &= ~CDPSTA_PUNCH;
    sim_fwrite(buff, 2, 80, uptr->fileref);
    return SCPE_OK;
}

uint32 cdp(UNIT * uptr, uint16 cmd, uint16 dev)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    int                 u = (uptr - cdp_unit);
    int                 i;

    if ((uptr->flags & UNIT_ATT) != 0 && cmd == OP_WRS) {
	/* Start device */
	if (!(uptr->u5 & CDPSTA_CMD)) {
	    dev_pulse[chan] &= ~PUNCH_M;
	    uptr->u5 &= ~CDPSTA_PUNCH;
	    if ((uptr->u5 & CDPSTA_ON) == 0) {
		uptr->wait = 330;	/* Startup delay */
	    } else if (uptr->u5 & CDPSTA_IDLE && uptr->wait <= 30) {
		uptr->wait += 85;	/* Wait for next latch point */
	    }
	    for (i = 0; i < 24; cdp_data[u].wbuff[i++] = 0) ;
	    uptr->u5 |= (CDPSTA_WRITE | CDPSTA_CMD);
	    uptr->u5 &= ~CDPSTA_POSMASK;
	    chan_set_sel(chan, 1);
	    chan_clear_status(chan);
	    sim_activate(uptr, 500);	/* activate */
	    sim_debug(DEBUG_CMD, &cdp_dev, "WRS unit=%d\n", u);
	    return 1;
	}
    }
    chan_set_attn(chan);
    return 0;
}

t_stat cdp_srv(UNIT * uptr)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    int                 u = (uptr - cdp_unit);
    int                 pos;

    /* Channel has disconnected, abort current card. */
    if (chan_status[chan] & DEV_DISCO && uptr->u5 & CDPSTA_CMD) {
	if ((uptr->u5 & CDPSTA_POSMASK) != 0)
	    punch_card(uptr, chan, u);
	uptr->u5 &= ~(CDPSTA_WRITE | CDPSTA_CMD | CDPSTA_POSMASK);
	chan_clear(chan, DEV_DISCO | DEV_WEOR | STA_SEL);
	sim_debug(DEBUG_CHAN, &cdp_dev, "unit=%d disconnect\n", u);
    }

    /* Check to see if we have timed out */
    if (uptr->wait != 0) {
	uptr->wait--;
	/* If at end of record and channel is still active, do another read */
	if (
	    ((uptr->
	      u5 & (CDPSTA_CMD | CDPSTA_IDLE | CDPSTA_WRITE | CDPSTA_ON))
	     == (CDPSTA_CMD | CDPSTA_IDLE | CDPSTA_ON)) && uptr->wait > 30
	    && (chan_status[chan] & STA_SEL)) {
	    uptr->u5 |= CDPSTA_WRITE;
	    uptr->u5 &= ~CDPSTA_IDLE;
	    chan_set(chan, DEV_WRITE);
	    chan_clear(chan, DEV_WEOR);
	    sim_debug(DEBUG_CHAN, &cdp_dev, "unit=%d restarting\n", u);
	}
	sim_activate(uptr, 500);	/* activate */
	return SCPE_OK;
    }

    /* If no write request, go to idle mode */
    if ((uptr->u5 & CDPSTA_WRITE) == 0) {
	if ((uptr->u5 & (CDPSTA_IDLE | CDPSTA_ON)) ==
	    (CDPSTA_IDLE | CDPSTA_ON)) {
	    uptr->wait = 85;	/* Delay 85ms */
	    uptr->u5 &= ~CDPSTA_IDLE;	/* Not running */
	    sim_activate(uptr, 500);
	} else {
	    uptr->u5 &= ~CDPSTA_ON;	/* Turn motor off */
	}
	return SCPE_OK;
    }

    /* Motor is up to speed now */
    uptr->u5 |= CDPSTA_ON;
    uptr->u5 &= ~CDPSTA_IDLE;	/* Not running */

    if (dev_pulse[chan] & PUNCH_M)
	uptr->u5 |= CDPSTA_PUNCH;

    pos = (uptr->u5 & CDPSTA_POSMASK) >> CDPSTA_POSSHIFT;
    sim_debug(DEBUG_DATA, &cdp_dev, "unit=%d write column %d ", u, pos);
    switch (chan_read(chan, &cdp_data[u].wbuff[pos], 0)) {
    case DATA_OK:
	sim_debug(DEBUG_DATA, &cdp_dev, " %012llo\n\r",
		  cdp_data[u].wbuff[pos]);
	pos++;
	if (pos != 24) {
	    uptr->wait = 0;
	    uptr->u5 &= ~CDPSTA_POSMASK;
	    uptr->u5 |= (pos << CDPSTA_POSSHIFT) & CDPSTA_POSMASK;
	    sim_activate(uptr, (pos & 1) ? 100 : 540);
	    return SCPE_OK;
	}
    case END_RECORD:
	sim_debug(DEBUG_DATA, &cdp_dev, "eor\n");
	chan_set(chan, DEV_REOR);
	chan_clear(chan, STA_WAIT);
	if (pos != 0)
	    punch_card(uptr, chan, u);
	if (pos == 24)
	    uptr->wait = 85;	/* Print wheel gap */
	else
	    uptr->wait = 8 * (12 - (pos / 2)) + 85;
	uptr->u5 |= CDPSTA_IDLE;
	uptr->u5 &= ~(CDPSTA_WRITE | CDPSTA_POSMASK);
	break;
    case TIME_ERROR:
	chan_set_attn(chan);
	sim_debug(DEBUG_DATA, &cdp_dev, "no data\n");
	chan_set(chan, DEV_REOR);
	chan_clear(chan, STA_WAIT);
	if (pos != 0)
	    punch_card(uptr, chan, u);
	if (pos == 24)
	    uptr->wait = 85;	/* Print wheel gap */
	else
	    uptr->wait = 8 * (12 - (pos / 2)) + 85;
	uptr->u5 |= CDPSTA_IDLE;
	uptr->u5 &= ~(CDPSTA_WRITE | CDPSTA_POSMASK);
	break;
    }

    sim_activate(uptr, 500);
    return SCPE_OK;
}

void
cdp_ini(UNIT * uptr, t_bool f)
{
    uptr->u5 = 0;
}

t_stat
cdp_reset(DEVICE * dptr)
{
    return SCPE_OK;
}

t_stat
cdp_attach(UNIT * uptr, char *file)
{
    t_stat              r;

    if ((r = attach_unit(uptr, file)) != SCPE_OK)
	return r;
    uptr->u5 = CDPSTA_POSMASK;
    return SCPE_OK;
}

t_stat
cdp_detach(UNIT * uptr)
{
    return detach_unit(uptr);
}
