/* i7090_cdr.c: IBM 7090 Card Read.

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   This is the standard card reader.

*/

#include "i7090_defs.h"

#define NUM_DEVS_CDR	4
#define UNIT_CDR	UNIT_ATTABLE | UNIT_RO | UNIT_DISABLE | UNIT_ROABLE

/* Flags for punch and reader. */
#define UNIT_V_MODE	(UNIT_V_LOCAL)
#define UNIT_MODE	(3 << UNIT_V_MODE)
#define MODE_AUTO	(0 << UNIT_V_MODE)
#define MODE_BIN	(1 << UNIT_V_MODE)
#define MODE_TEXT	(2 << UNIT_V_MODE)

/* std devices. data structures

   chan_dev	Channel device descriptor
   chan_unit	Channel unit descriptor
   chan_reg	Channel register list
   chan_mod	Channel modifiers list
*/

/* Device status information stored in u5 */
#define CDRSTA_READ	000001	/* Unit is in read */
#define CDRSTA_WRITE	000002	/* Unit is in write */
#define CDRSTA_ON	000004	/* Unit is running */
#define CDRSTA_EOF	000010	/* Hit end of file */
#define CDRSTA_EOR	000020	/* Hit end of record */
#define CDRSTA_IDLE	000040	/* Unit between operation */
#define CDRSTA_CMD	000100	/* Unit has recieved a cmd */
#define CDRPOSMASK      077000	/* Bit Mask to retrive drum position */
#define CDRPOSSHIFT	9

struct _cdr_data
{
    uint8               ptr;	/* Pointer in buffer */
    t_uint64            wbuff[24];	/* Card buffer */
    uint16              cbuff[80];	/* Image buffer */
}
cdr_data[NUM_DEVS_CDR];

uint32              cdr(UNIT *, uint16, uint16);
t_stat              cdr_srv(UNIT *);
t_stat              cdr_boot(int32, DEVICE *);
t_stat              cdr_reset(DEVICE *);
t_stat              cdr_attach(UNIT *, char *);
t_stat              cdr_detach(UNIT *);
extern t_stat       chan_boot(int32, DEVICE *);
extern char         ascii_to_six[128];
extern char         six_to_ascii[64];
extern uint8        iocheck;

DIB                 cdr_dib = { CHAN_7607, 1, 0321, 0777, &cdr, NULL };

UNIT                cdr_unit[] = {
    {UDATA(&cdr_srv, UNIT_S_CHAN(1) | UNIT_CDR, 0), 3000},	/* A */
    {UDATA(&cdr_srv, UNIT_S_CHAN(2) | UNIT_CDR, 0), 3000},	/* B */
    {UDATA(&cdr_srv, UNIT_S_CHAN(3) | UNIT_CDR | UNIT_DIS, 0), 3000},	/* C */
    {UDATA(&cdr_srv, UNIT_S_CHAN(0) | UNIT_CDR | UNIT_DIS, 0), 3000},	/* D */
};

MTAB                cdr_mod[] = {
    {UNIT_MODE, MODE_AUTO, "AUTO", "AUTO", NULL, NULL, NULL},
    {UNIT_MODE, MODE_BIN, "BIN", "BIN", NULL, NULL, NULL},
    {UNIT_MODE, MODE_TEXT, "TEXT", "TEXT", NULL, NULL, NULL},
    
	{MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "CHAN", "CHAN", &set_chan,
     &get_chan, NULL},
    {0}
};

DEVICE              cdr_dev = {
    "CR", cdr_unit, NULL, cdr_mod,
    NUM_DEVS_CDR, 8, 15, 1, 8, 36,
    NULL, NULL, &cdr_reset, &cdr_boot, &cdr_attach, &cdr_detach,
    &cdr_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};

t_stat
read_card(UNIT * uptr, int chan, int unit)
{
    uint16              cbuff[80];	/* Temp conversion buffer */
    uint16             *buff;	/* Pointer to buffer */
    int                 i;
    char                c;
    uint16              temp;
    int                 len;
    int                 start = 0;
    int                 size;
    int                 col;

    if ((uptr->flags & UNIT_ATT) == 0)
	return SCPE_UNATT;	/* attached? */
    /* On eof, just return */
    if (uptr->u5 & CDRSTA_EOF)
	return SCPE_OK;		/* At EOF? */

    sim_debug(DEBUG_DETAIL, &cdr_dev, "%d: Read Card\n", unit);
/* Move data to start at begining of buffer */
    if (uptr->u4 > 0) {
	int                 ptr = uptr->u4;

	while (ptr < 160)
	    ((char *) cdr_data[unit].cbuff)[start++] =
		((char *) cdr_data[unit].cbuff)[ptr++];
	if (!feof(uptr->fileref))
	    len = sim_fread(&((char *) cdr_data[unit].cbuff)[start], 1,
			    160 - start, uptr->fileref);
	else
	    len = 0;
	size = len + start;
    } else {
/* Load rest of buffer */
	len = sim_fread(&cdr_data[unit].cbuff[0], 2, 80, uptr->fileref);
	size = len * 2;
    }
    if ((len < 0 || size == 0) && feof(uptr->fileref)) {
	uptr->u5 |= CDRSTA_EOF;
	return SCPE_OK;
    }
    if (ferror(uptr->fileref)) {	/* error? */
	perror("Card reader I/O error");
	clearerr(uptr->fileref);
	iocheck = 1;
	return SCPE_IOERR;
    }

    /* Check buffer to see if binary card in it. */
    for (i = 0, temp = 0; i < 80; temp |= cdr_data[unit].cbuff[i++]) ;
    if ((size == 160 && (temp & 0xf000) == 0) ||
	(uptr->flags & UNIT_MODE) == MODE_BIN) {
	/* Probably bin card */
	if ((uptr->flags & UNIT_MODE) == MODE_TEXT)
	    iocheck = 1;
	buff = &cdr_data[unit].cbuff[0];
	uptr->u4 = 0;
    } else {
	/* Probably ascii card */
	if ((uptr->flags & UNIT_MODE) == MODE_BIN)
	    iocheck = 1;

	/* Clear input buffer */
	for (col = 0; col < 80; cbuff[col++] = 0) ;
	/* Convert text line into card image */
	for (col = 0, i = 0; col < 80 && i < size; i++) {
	    c = ((char *) cdr_data[unit].cbuff)[i];
	    switch (c) {
	    case '\0':
	    case '\r':
		break;		/* Ignore these */
	    case '\t':
		col = (col | 7) + 1;	/* Mult of 10 */
		break;
	    case '~':		/* End of file mark */
		cbuff[col++] = 0x006;
		break;
	    case '\n':
		col = 80;
		break;
	    default:
		c = ascii_to_six[0177 & c];
		if (c != -1) {
		    uint16              t;

		    /* Fix mess up in codes */
		    if (c == 060)
			c = 0;
		    else if (c == 0)
			c = 060;

		    /* Convert to top column */
		    switch (c & 060) {
		    case 000:
			t = 0x000;
			break;
		    case 020:
			t = 0x800;
			break;
		    case 040:
			t = 0x400;
			break;
		    case 060:
			t = 0x200;
			break;
		    }

		    /* Convert to 0-9 row */
		    c &= 017;
		    if (c > 9) {
			t |= 0x2;	/* Col 8 */
			c -= 8;
		    }
		    if (c != 0)
			t |= 1 << (9 - c);
		    cbuff[col++] = t;
		}
		/* Eat cr if line exactly 80 columns */
		if (col == 80) {
		    c = ((char *) cdr_data[unit].cbuff)[i + 1];
		    if (c == '\n')
			i++;
		}
	    }
	}
	buff = &cbuff[0];
	if (i < size)
	    uptr->u4 = i;
	else
	    uptr->u4 = 0;
    }
    /* Bit flip into read buffer */
    for (i = 0; i < 24; i++) {
	int                 bit = 1 << (i / 2);
	t_uint64            mask = 1;
	t_uint64            wd = 0;
	int                 b = 36 * (i & 1);

	for (col = 35; col >= 0; mask <<= 1) {
	    if (buff[col-- + b] & bit)
		wd |= mask;
	}
	cdr_data[unit].wbuff[i] = wd;
    }
    return SCPE_OK;
}

uint32 cdr(UNIT * uptr, uint16 cmd, uint16 dev)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);

    if ((uptr->flags & UNIT_ATT) != 0 && cmd == OP_RDS) {
	int                 u = (uptr - cdr_unit);

	/* Start device */
	if ((uptr->u5 & CDRSTA_CMD) == 0) {
	    if ((uptr->u5 & (CDRSTA_ON | CDRSTA_IDLE) ==
		 (CDRSTA_ON | CDRSTA_IDLE)) && (uptr->wait <= 60)) {
		uptr->wait += 100;	/* Wait for next latch point */
	    } else
		uptr->wait = 450;	/* Startup delay */
	    uptr->u5 |= CDRSTA_READ | CDRSTA_CMD | CDRPOSMASK;
	    chan_set_sel(chan, 0);
	    chan_clear_status(chan);
	    sim_activate(uptr, 10);	/* activate */
	    sim_debug(DEBUG_CMD, &cdr_dev, "RDS unit=%d\n", u);
	    return 1;
	}
	return 0;
    }
    chan_set_attn(chan);
    return -1;
}

t_stat cdr_srv(UNIT * uptr)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    int                 u = (uptr - cdr_unit);
    int                 pos;

    /* Channel has disconnected, abort current read. */
    if (chan_status[chan] & DEV_DISCO && uptr->u5 & CDRSTA_CMD) {
	uptr->u5 &= ~(CDRSTA_READ | CDRSTA_CMD);
	uptr->u5 |= CDRPOSMASK;
	chan_clear(chan, DEV_DISCO | DEV_WEOR | STA_SEL);
	sim_debug(DEBUG_CHAN, &cdr_dev, "unit=%d disconnecting\n", u);
    }

    /* Check to see if we have timed out */
    if (uptr->wait != 0) {
	/* If at end of record and channel is still active, do another read */
	if (uptr->wait == 30
	    && ((uptr->u5 & (CDRSTA_CMD|CDRSTA_IDLE|CDRSTA_READ|CDRSTA_ON))
		 == (CDRSTA_CMD | CDRSTA_IDLE | CDRSTA_ON))
	    && (chan_status[chan] & STA_ACTIVE)) {
	    uptr->u5 |= CDRSTA_READ;
	    sim_debug(DEBUG_CHAN, &cdr_dev, "unit=%d restarting\n", u);
	}
	uptr->wait--;
	sim_activate(uptr, 10);	/* activate */
	return SCPE_OK;
    }

    /* If no read request, go to idle mode */
    if ((uptr->u5 & CDRSTA_READ) == 0) {
	if ((uptr->u5 & CDRSTA_EOF) || (uptr->u5 & CDRSTA_IDLE)) {
	    uptr->u5 &= ~(CDRSTA_ON | CDRSTA_IDLE);	/* Turn motor off */
	} else {
	    uptr->wait = 120;	/* Delay 85ms */
	    uptr->u5 |= CDRSTA_IDLE;	/* Go idle */
	    sim_activate(uptr, 10);
	}
	return SCPE_OK;
    }

    /* Motor is up to speed now */
    uptr->u5 |= CDRSTA_ON;
    uptr->u5 &= ~CDRSTA_IDLE;

    pos = (uptr->u5 & CDRPOSMASK) >> CDRPOSSHIFT;
    if (pos == (CDRPOSMASK >> CDRPOSSHIFT)) {
	if (read_card(uptr, chan, u) != SCPE_OK) {
	    sim_debug(DEBUG_EXP, &cdr_dev, "unit=%d Setting ATTN\n", u);
	    chan_set_error(chan);
	    chan_set_attn(chan);
	    uptr->u5 &= ~CDRSTA_READ;
	    sim_activate(uptr, 25);
	    return SCPE_OK;
	}
	if (uptr->u5 & CDRSTA_EOF) {
	    chan_set_eof(chan);
	    uptr->u5 &= ~CDRSTA_READ;
	    sim_activate(uptr, 25);
	    return SCPE_OK;
	}
	pos = 0;
    }

    switch (chan_write
	    (chan, &cdr_data[u].wbuff[pos], (pos == 23) ? DEV_REOR : 0)) {
    case DATA_OK:
	sim_debug(DEBUG_DATA, &cdr_dev, "unit=%d read row %d %012llo\n", u,
		  pos, cdr_data[u].wbuff[pos]);
	pos++;
	uptr->u5 &= ~CDRPOSMASK;
	uptr->u5 |= pos << CDRPOSSHIFT;
	if (pos != 24) {
	    uptr->wait = 0;
	    sim_activate(uptr, (pos & 1) ? 45 : 650);
	    return SCPE_OK;
	}

    case END_RECORD:
	sim_debug(DEBUG_CHAN, &cdr_dev, "unit=%d got EOR\n", u);
	uptr->u5 &= ~CDRSTA_READ;
	uptr->u5 |= CDRSTA_EOR | CDRPOSMASK;
	chan_set(chan, DEV_REOR);
	uptr->wait = 5 * (12 - (pos / 2)) + 120;
	break;

    case TIME_ERROR:
	sim_debug(DEBUG_EXP, &cdr_dev, "unit=%d no data\n", u);
	uptr->u5 &= ~CDRSTA_READ;
	uptr->u5 |= CDRSTA_EOR | CDRPOSMASK;
	uptr->wait = 5 * (12 - (pos / 2)) + 100;
	break;
    }

    sim_activate(uptr, 10);
    return SCPE_OK;
}

/* Boot from given device */
t_stat
cdr_boot(int32 unit_num, DEVICE * dptr)
{
    UNIT               *uptr = &dptr->units[unit_num];
    int                 chan = UNIT_G_CHAN(uptr->flags);
    t_stat              r;

    if ((uptr->flags & UNIT_ATT) == 0)
	return SCPE_UNATT;	/* attached? */
    uptr->u5 = 0;
/* Init for a read */
    if (cdr(uptr, OP_RDS, cdr_dib.addr) != 1)
	return STOP_IONRDY;
    r = read_card(uptr, chan, unit_num);
    if (r != SCPE_OK)
	return r;
/* Copy first three records. */
    uptr->u5 &= ~CDRPOSMASK;
    M[0] = cdr_data[unit_num].wbuff[0];
    M[1] = cdr_data[unit_num].wbuff[1];
    if (chan != 0) {
	M[2] = cdr_data[unit_num].wbuff[2];
	uptr->u5 |= 3 << CDRPOSSHIFT;
    } else {
	uptr->u5 |= 2 << CDRPOSSHIFT;
    }
/* Make sure channel is set to start reading rest. */
    return chan_boot(unit_num, dptr);
}

t_stat
cdr_reset(DEVICE * dptr)
{
    return SCPE_OK;
}

t_stat
cdr_attach(UNIT * uptr, char *file)
{
    t_stat              r;

    if ((r = attach_unit(uptr, file)) != SCPE_OK)
	return r;
    uptr->u5 &= ~CDRSTA_EOF;
    uptr->u4 = 0;
    return SCPE_OK;
}

t_stat
cdr_detach(UNIT * uptr)
{
    return detach_unit(uptr);
}
