/* i7090_chan.c: IBM 7090 Channel simulator

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   channel		

   The system state for the IBM 7090 channel is:

   AC<1:35,P,Q,S>	AC register
   MQ<1:35,S>		MQ register
   XR<0:15>[7]		XR index registers register
   IC<0:15>		program counter
*/

#include "i7090_defs.h"

extern DEVICE      *sim_devices[];
extern uint16       iotraps;
extern uint8        iocheck;
extern uint8        dualcore;
extern uint32       sim_interval;
extern UNIT         cpu_unit;
extern uint16       IC;
extern uint32       drum_addr;
extern uint32       hsdrm_addr;

#define UNIT_V_MODE	(UNIT_V_MODEL + 2)
#define UNIT_V_SET	(UNIT_V_MODE + 1)
#define CHAN_MODE	(1 << UNIT_V_MODE)
#define CHAN_SET	(1 << UNIT_V_SET)

t_stat              chan_reset(DEVICE * dptr);
void                chan_fetch(int chan);

/* Channel data structures

   chan_dev	Channel device descriptor
   chan_unit	Channel unit descriptor
   chan_reg	Channel register list
   chan_mod	Channel modifiers list
*/

uint16              caddr[NUM_CHAN];		/* Channel memory address */
uint8               cmd[NUM_CHAN];		/* Current command */
uint16              wcount[NUM_CHAN];		/* Word count */
t_uint64            assembly[NUM_CHAN];		/* Assembly register */
uint16              location[NUM_CHAN];		/* Pointer to next opcode */
uint16              chan_status[NUM_CHAN];	/* Unit status */
uint16              chan_info[NUM_CHAN];	/* Private channel info */
uint8               counter[NUM_CHAN];		/* Channel counter */
uint8               sms[NUM_CHAN];		/* Channel mode infomation */
uint16              chan_sense[NUM_CHAN];	/* Sense info for 7909 channels */
uint8               chan_irq[NUM_CHAN];		/* Channel has a irq pending */
uint16              num_devs[NUM_CHAN];		/* Number of devices on channel */

/* 7607 channel commands */
#define IOCD	000
#define TCH	010
#define IORP	020
#define IORT	030
#define IOCP	040
#define IOCT	050
#define IOSP	060
#define IOST	070

/* 7909 channel commands */
#define WTR     000
#define WTRX    004
#define XMT	001
#define XMTX	005
#define TCH	010
#define TCHX	014
#define LIPT	011
#define LIPTX	015
#define CTL 	020
#define CTLR	021
#define CTLW	024
#define SNS	025
#define LAR     030
#define SAR     031
#define TWT     034
#define XXXX    035
#define CPYP	040
#define CPYP2	041
#define CPYP3	044
#define CPYP4	045
#define CPYD	050
#define TCM	051
#define CPYDX	054
#define TCMX	055
#define XXXZ    060
#define LIP     061
#define TDC     064
#define LCC     065
#define SMS     070
#define ICC	071
#define ICCX	075

#define READ_WRD	1
#define WRITE_WRD	2

/* Values for chan_info */
#define CHAINF_START	1	/* Channel started */
#define CHAINF_RUN	2	/* Transfer in progress */

UNIT                chan_unit[] = {
    /* Puesdo channel for 704 devices */
    {UDATA(NULL, UNIT_DISABLE | CHAN_SET | CHAN_S_TYPE(CHAN_704), 0)},
    /* Normal channels */
    {UDATA(NULL, CHAN_MODE | CHAN_SET | CHAN_S_TYPE(CHAN_7607), 0)},	/* A */
    {UDATA(NULL, UNIT_DISABLE | CHAN_MODE, 0)},	/* B */
    {UDATA(NULL, UNIT_DISABLE | CHAN_MODE, 0)},	/* C */
    {UDATA(NULL, UNIT_DISABLE | CHAN_MODE, 0)},	/* D */
    {UDATA(NULL, UNIT_DISABLE | CHAN_MODE, 0)},	/* E */
    {UDATA(NULL, UNIT_DISABLE | CHAN_MODE, 0)},	/* F */
    {UDATA(NULL, UNIT_DISABLE | CHAN_MODE, 0)},	/* G */
    {UDATA(NULL, UNIT_DISABLE | CHAN_MODE, 0)}	/* H */
};

REG                 chan_reg[] = {
    {BRDATA(ADDR, caddr, 8, 16, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(CMD, cmd, 8, 6, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(WC, wcount, 8, 15, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(ASM, assembly, 8, 36, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(LOCATION, location, 8, 16, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(STATUS, chan_status, 2, 16, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(SENSE, chan_status, 2, 16, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(COUNTER, counter, 8, 6, NUM_CHAN), REG_RO | REG_UNIT},
    {BRDATA(SMS,    counter, 2, 6, NUM_CHAN), REG_RO | REG_UNIT},
    {NULL}
};

MTAB                chan_mod[] = {
    {CHAN_MODEL, CHAN_S_TYPE(CHAN_7607), "7607", "7607", NULL, NULL, NULL},
    {CHAN_MODEL, CHAN_S_TYPE(CHAN_7909), "7909", "7909", NULL, NULL, NULL},
    {CHAN_MODEL, CHAN_S_TYPE(CHAN_7289), "7289", "7289", NULL, NULL, NULL},
    {CHAN_MODEL, CHAN_S_TYPE(CHAN_704), "704 Channel", NULL, NULL, NULL, NULL},
    {CHAN_MODE, 0, "FIXED", "FIXED", NULL, NULL, NULL},
    {CHAN_MODE, CHAN_MODE, "AUTO", "AUTO", NULL, NULL, NULL},
    {CHAN_SET, CHAN_SET, "set", NULL, NULL, NULL, NULL},
    {0}
};

/* Simulator debug controls */
DEBTAB              chn_debug[] = {
    {"CHANNEL", DEBUG_CHAN},
    {"TRAP", DEBUG_TRAP},
    {"CMD", DEBUG_CMD},
    {"DATA", DEBUG_DATA},
    {"DETAIL", DEBUG_DETAIL},
    {"EXP", DEBUG_EXP},
    {"SENSE", DEBUG_SNS},
    {"CH0", 0x0100 << 0},
    {"CHA", 0x0100 << 1},
    {"CHB", 0x0100 << 2},
    {"CHC", 0x0100 << 3},
    {"CHD", 0x0100 << 4},
    {"CHE", 0x0100 << 5},
    {"CHF", 0x0100 << 6},
    {"CHG", 0x0100 << 7},
    {"CHH", 0x0100 << 8},
    {0, 0}
};

DEVICE              chan_dev = {
    "CH", chan_unit, chan_reg, chan_mod,
    NUM_CHAN, 8, 15, 1, 8, 36,
    NULL, NULL, &chan_reset, NULL, NULL, NULL,
    NULL, DEV_DEBUG, 0, chn_debug
};

t_stat
chan_reset(DEVICE * dptr)
{
    int                 i;

    /* Clear channel assignment */
    for (i = 0; i < NUM_CHAN; i++) {
	if (chan_unit[i].flags & CHAN_MODE)
	    chan_unit[i].flags &= ~CHAN_SET;
	else
	    chan_unit[i].flags |= CHAN_SET;
	chan_status[i] = 0;
	chan_info[i] = 0;
	chan_sense[i] = 0;
	caddr[i] = 0;
	cmd[i] = 0;
	sms[i] = 0;
	chan_irq[i] = 0;
	wcount[i] = 0;
	location[i] = 0;
	counter[i] = 0;
	num_devs[i] = 0;
    }
    /* Build channel array */
    for (i = 0; sim_devices[i] != NULL; i++) {
	UNIT               *uptr = sim_devices[i]->units;
	DIB                *dibp = (DIB *) sim_devices[i]->ctxt;
	int                 num;

	/* If no DIB, not channel device */
	if (dibp == NULL)
	    continue;
	for (num = sim_devices[i]->numunits; num > 0; num--) {
	    int                 chan = UNIT_G_CHAN(uptr->flags);
	    int                 type = CHAN_G_TYPE(chan_unit[chan].flags);

	    if ((uptr->flags & UNIT_DIS) == 0) {
		int                 ctype = dibp->ctype;

		if (ctype < CHAN_7289) {
		    if (chan == 0)
			ctype = CHAN_704;
		    else
			ctype = CHAN_7607;
		}

		if (ctype != type) {
		    if ((chan_unit[chan].flags & CHAN_SET) ||
			((chan_unit[chan].flags & CHAN_MODE)
			 && num_devs[chan] != 0)) {
			uptr->flags |= UNIT_DIS;
			goto next;
		    } else {
			chan_unit[chan].flags &= ~CHAN_MODEL;
			chan_unit[chan].flags |= CHAN_S_TYPE(ctype);
		    }
		}
		chan_unit[chan].flags |= CHAN_SET;
		num_devs[chan]++;
		if (dibp->ini != NULL)
		    dibp->ini(uptr, 1);
	    }
	  next:
	    uptr++;
	}
    }
    return SCPE_OK;
}

/* Boot from given device */
t_stat
chan_boot(int32 unit_num, DEVICE * dptr)
{
    /* Tell device to do a read, 3 records */
    /* Set channel address = 0, wc = 3, location = 0, CMD=0 */
    /* Set M[1] = TCO? 1, IC = 1 */
    UNIT               *uptr = &dptr->units[unit_num];
    int                 chan = UNIT_G_CHAN(uptr->flags);

    if (chan == 0) {
	IC = 0;
    } else {
	IC = 1;
	/* Grab next channel command */
	location[chan] = 0;
	chan_fetch(chan);
    }
    chan_status[chan] |= STA_ACTIVE;
    chan_status[chan] &= ~STA_PEND;
    return SCPE_OK;
}

/* Preform BCD to binary translation for 7909 channel */
void
bcd_xlat(int chan, int direction)
{
    int                 i;
    t_uint64            na = 0;

    for (i = 30; i >= 0; i -= 6) {
	uint8               ch = (uint8)(assembly[chan] >> i) & 077;

	if (direction) {	/* D->M Read */
	    switch (ch & 060) {
	    case 000:
		if (ch == 0)
		    ch = 060;
		else if (ch == 012)
		    ch = 0;
		break;
	    case 020:
		 if (ch == 020)
                     ch = 052;   /* => 012 */
	    case 060:	
		ch ^= 040;
	    case 040:
		break;
	    }
	} else {		/* M->D Write */
	    switch (ch & 060) {
	    case 000:
		if (ch == 0)
		    ch = 012;
		else if (ch == 012)
		    ch = 020;
		break;
	    case 060:
		if (ch == 060)
		    ch = 040;	/* => 000 */
	    case 020:
		ch ^= 040;
	    case 040:
		break;
	    }
	}
	na |= ((t_uint64) ch) << i;
    }
    assembly[chan] = na;
}

/* Execute the next channel instruction. */
void
chan_proc()
{
    int                 chan;
    int                 cmask;

    /* Scan channels looking for work */
    for (chan = 0; chan < NUM_CHAN; chan++) {
	/* Skip if channel is disabled */
	if (chan_unit[chan].flags & UNIT_DIS)
	    continue;

	cmask = 0x0100 << chan;
	switch (CHAN_G_TYPE(chan_unit[chan].flags)) {
	case CHAN_7289:	/* Special channel for HS drum */
	    /* On first command, copy it to drum address and load another */
	    if ((chan_info[chan] & (CHAINF_RUN | CHAINF_START)) ==
		CHAINF_START) {
		hsdrm_addr = (int)M[location[chan] - 1];
		chan_info[chan] |= CHAINF_RUN;
		chan_fetch(chan);
		continue;
	    }
	    if ((chan_info[chan] & CHAINF_START) == 0)
		continue;
	    /* Fall through and behave like 7607 from now on */
	case CHAN_7607:
	    /* If channel is disconnecting, do nothing */
	    if (chan_status[chan] & DEV_DISCO)
		continue;

	    /* If device requested attention, abort current command */
	    if (chan_status[chan] & CHS_ATTN) {
		if (chan_dev.dctrl & cmask)
		    sim_debug(DEBUG_EXP, &chan_dev, "chan %d Attn\n",
			      chan);
		chan_status[chan] &=
		    ~(CHS_ATTN | STA_START | STA_ACTIVE | STA_WAIT);
		chan_info[chan] = 0;
		chan_status[chan] |= (DEV_DISCO);
		continue;
	    }

	    /* If no select, stop channel */
	    if ((chan_status[chan] & STA_SEL) == 0
		&& (chan_status[chan] & STA_TWAIT)) {
		if (chan_dev.dctrl & cmask)
		    sim_debug(DEBUG_TRAP, &chan_dev, "chan %d Trap\n",
			      chan);
		iotraps |= 1 << chan;
		chan_status[chan] &=
		    ~(STA_START | STA_ACTIVE | STA_WAIT | STA_TWAIT);
		chan_info[chan] = 0;
		continue;
	    }

	    /* No activity, move along */
	    if ((chan_status[chan] & (STA_ACTIVE | STA_WAIT)) == 0)
		continue;

	    /* If command is a transfer, Do transfer */
	    while ((cmd[chan] & 070) == TCH) {
		location[chan] = caddr[chan];
		chan_fetch(chan);
	    }

	    /* None disabled, active channel is if transfering */
	    switch (chan_status[chan] & (DEV_WRITE | DEV_FULL)) {
		/* Device has given us a dataword */
	    case DEV_FULL:
		/* If we are not waiting EOR save it in memory */
		if ((chan_status[chan] & STA_WAIT) == 0) {
		    if (chan_dev.dctrl & cmask)
			sim_debug(DEBUG_DATA, &chan_dev,
				  "chan %d data < %012llo\n",
				  chan, assembly[chan]);
		    if ((cmd[chan] & 1) == 0)
			M[caddr[chan]] = assembly[chan];
		    if (dualcore)
			caddr[chan] =
			    (0100000 & caddr[chan]) | ((caddr[chan] + 1) &
						       (MEMSIZE - 1));
		    else
			caddr[chan] = (caddr[chan] + 1) & (MEMSIZE - 1);
		    wcount[chan]--;
		}
		chan_status[chan] &= ~DEV_FULL;

		/* Device does not need a word and has not given us one */
	    case 0:
		/* Device idle, expecting data from it */
		/* Check if got EOR */
		if (chan_status[chan] & DEV_REOR) {
		    if (chan_dev.dctrl & cmask)
			sim_debug(DEBUG_EXP, &chan_dev, "chan %d EOR %o\n",
				  chan, cmd[chan] & 070);
		    switch (cmd[chan] & 070) {
		    case IORP:
			chan_status[chan] &= ~(DEV_REOR | STA_WAIT);
		    case IOSP:
			chan_fetch(chan);
			chan_status[chan] |= STA_ACTIVE;
			continue;	/* Handle new command next time */
		    case IORT:
			chan_status[chan] &= ~(STA_WAIT | DEV_REOR);
		    case IOST:
			chan_status[chan] &= ~(STA_ACTIVE);
			if ((chan_status[chan] & STA_PEND) == 0) {
			    chan_status[chan] |= DEV_DISCO | STA_TWAIT;
			    chan_status[chan] &= ~(STA_START);
			}
			continue;
		    }
		}
		if (wcount[chan] == 0
		    && (chan_status[chan] & STA_WAIT) == 0) {
		    if (chan_dev.dctrl & cmask)
			sim_debug(DEBUG_EXP, &chan_dev,
				  "chan %d < WC0 %o\n", chan,
				  cmd[chan] & 070);
		    switch (cmd[chan] & 070) {
		    case IOCD:	/* Transfer and disconnect */
			chan_status[chan] |= DEV_DISCO | DEV_WEOR;
			chan_status[chan] &=
			    ~(STA_START | STA_ACTIVE | STA_PEND);
			if (CHAN_G_TYPE(chan_unit[chan].flags) ==
			    CHAN_7289) iotraps |= 1 << chan;
			chan_info[chan] = 0;
			break;

		    case IORP:	/* Transfer until end of record */
			chan_status[chan] |= STA_WAIT | DEV_WEOR;
			break;
		    case IOSP:	/* Transfer and proceed */
		    case IOCP:	/* Transfer and proceed, no eor */
			chan_fetch(chan);
			break;

		    case IORT:	/* Transfer, continue if LCH pending, */
			/* else trap, Skip rest of record */
			chan_status[chan] |=
			    STA_WAIT | DEV_WEOR | STA_TWAIT;
			break;
		    case IOST:	/* Transfer, continue if LCH, else trap */
		    case IOCT:	/* Transfer but no end of record, else trap */
			chan_status[chan] &= ~STA_ACTIVE;
			if ((chan_status[chan] & STA_PEND) == 0) {
			    chan_status[chan] |= DEV_DISCO | STA_TWAIT;
			    chan_status[chan] &= ~(STA_START | STA_WAIT);
			}
			break;
		    }
		}
		break;

		/* Device has word, but has not taken it yet */
	    case DEV_WRITE | DEV_FULL:
		continue;	/* Do nothing if no data xfer pending */

		/* Device needs a word of data */
	    case DEV_WRITE:	/* Device needs data word */
		/* Wait for device to recognize EOR */
		if (chan_status[chan] & DEV_WEOR)
		    continue;
		/* Give device new word if we have one */
		if (wcount[chan] != 0) {
		    if (cmd[chan] & 1)	// || (chan_status[chan] & STA_WAIT))
			assembly[chan] = 0;
		    else {
			assembly[chan] = M[caddr[chan]];
			if (chan_dev.dctrl & cmask)
			    sim_debug(DEBUG_DATA, &chan_dev,
				      "chan %d data > %012llo\n", chan,
				      assembly[chan]);
			if (dualcore)
			    caddr[chan] =
				(0100000 & caddr[chan]) |
				((caddr[chan] + 1) & (MEMSIZE - 1));
			else
			    caddr[chan] =
				(caddr[chan] + 1) & (MEMSIZE - 1);
			wcount[chan]--;
		    }
		    if ((chan_status[chan] & STA_WAIT) == 0) {
			chan_status[chan] |= DEV_FULL;
		    }
		    continue;	/* Don't start next command until data taken */
		}

		/* Get here if wcount == 0 */
		if (chan_dev.dctrl & cmask)
		    sim_debug(DEBUG_EXP, &chan_dev, "chan %d > WC0 %o\n",
			      chan, cmd[chan] & 070);

		switch (cmd[chan] & 070) {
		case IOCD:	/* Transfer and disconnect */
		    chan_status[chan] |= DEV_DISCO | DEV_WEOR;
		    chan_status[chan] &=
			~(STA_START | STA_ACTIVE | STA_PEND);
		    if (CHAN_G_TYPE(chan_unit[chan].flags) == CHAN_7289)
			iotraps |= 1 << chan;
		    chan_info[chan] = 0;
		    if (chan_dev.dctrl & cmask)
			sim_debug(DEBUG_EXP, &chan_dev,
				  "chan %d > DISCO\n", chan);
		    break;

		case IORP:	/* Transfer until end of record */
		    chan_status[chan] |= DEV_WEOR;
		case IOSP:	/* Transfer and proceed */
		case IOCP:	/* Transfer and proceed, no eor */
		    chan_fetch(chan);
		    break;

		case IORT:	/* Transfer, continue if LCH pending, */
		    /* else trap, Skip rest of record */
		    chan_status[chan] |= DEV_WEOR;
		case IOST:	/* Transfer, continue if LCH, else trap */
		case IOCT:	/* Transfer but no end of record, else trap */
		    chan_status[chan] &= ~STA_ACTIVE;
		    if ((chan_status[chan] & STA_PEND) == 0) {
			chan_status[chan] |= DEV_DISCO | STA_TWAIT;
			chan_status[chan] &= ~(STA_START | STA_WAIT);
		    }
		    break;
		}
	    }
	    break;
	case CHAN_704:
	    if (chan_status[chan] & CHS_ATTN) {
		chan_status[chan] &=
		    ~(CHS_ATTN | STA_START | STA_ACTIVE | STA_WAIT);
		chan_status[chan] |= (DEV_DISCO);
	    }
	    break;
	case CHAN_7909:
	  again:
	    /* If channel is disconnecting, just hold on */
	    if (chan_status[chan] & DEV_DISCO)
		continue;

	    /* Nothing more to do if not active. */
	    if (chan_status[chan] & STA_ACTIVE) {
		/* Execute the next command */
		switch (cmd[chan]) {
		case XXXZ:
		case XXXX:
		case TWT:
		    /* Check if command not allowed */
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    if (chan_dev.dctrl & cmask)
			sim_debug(DEBUG_TRAP, &chan_dev, "chan %d Trap\n",
				  chan);
		    iotraps |= 1 << chan;
		    chan_sense[chan] |= CTL_INHB;
		case WTR:
		case WTRX:
		    /* Check if command not allowed */
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    /* Go into a wait state */
		    chan_status[chan] &= ~STA_ACTIVE;
		    location[chan]--;
		    break;
		case XMT:
		case XMTX:
		    /* Check if command not allowed */
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    if (wcount[chan] == 0)
			break;
		    wcount[chan]--;
		    M[caddr[chan]] = M[location[chan]];
		    if (dualcore)
			caddr[chan] =
			    (0100000 & caddr[chan]) | ((caddr[chan] + 1) &
						       (MEMSIZE - 1));
		    else
			caddr[chan] = (caddr[chan] + 1) & (MEMSIZE - 1);
		    location[chan]++;
		    continue;
		case LIPT:
		case LIPTX:
		    chan_sense[chan] &= ~(CTL_INHB | SNS_IRQS);
		    chan_irq[chan] = 0;
		case TCH:
		case TCHX:
		    location[chan] = caddr[chan];
		    break;
		case LIP:
		    chan_sense[chan] &= ~(CTL_INHB | SNS_IRQS);
		    chan_irq[chan] = 0;
		    location[chan] = (uint16)M[040 + (2 * chan)] & (MEMSIZE - 1);
		    break;
		case CTL:
		    if (chan_sense[chan] & CTL_CNTL)
			goto xfer;
		    if (chan_sense[chan] &
			(SNS_READ | SNS_WRITE | CTL_SNS1 | CTL_SNS2)) {
			chan9_seqcheck(chan);
			continue;
		    }
		    chan_sense[chan] |= CTL_CNTL;
		    goto finddev;
		case CTLR:
		    if (chan_sense[chan] & CTL_CNTL)
			goto xfer;
		    if (chan_sense[chan] &
			(SNS_READ | SNS_WRITE | CTL_SNS1 | CTL_SNS2)) {
			chan9_seqcheck(chan);
			break;
		    }
		    chan_sense[chan] |= CTL_CNTL | SNS_PREAD;
		    goto finddev;
		case CTLW:
		    if (chan_sense[chan] & CTL_CNTL)
			goto xfer;
		    if (chan_sense[chan] &
			(SNS_READ | SNS_WRITE | CTL_SNS1 | CTL_SNS2)) {
			chan9_seqcheck(chan);
			break;
		    }
		    chan_sense[chan] |= CTL_CNTL | SNS_PWRITE;
		    goto finddev;
		case SNS:
		    if (chan_sense[chan] &
			(CTL_CNTL | SNS_READ | SNS_WRITE)) {
			chan9_seqcheck(chan);
			break;
		    }
		    chan_sense[chan] |= CTL_SNS1;
		  finddev:
		    {
			DEVICE            **dptr;
			UNIT               *uptr;
			DIB                *dibp;

			for (dptr = sim_devices; *dptr != NULL; dptr++) {
			    int                 num = (*dptr)->numunits;
			    int                 j;

			    dibp = (DIB *) (*dptr)->ctxt;
			    /* If not device or 7909 type, just skip */
			    if (dibp == 0 || dibp->ctype != CHAN_7909)
				continue;
			    uptr = (*dptr)->units;
			    for (j = 0; j < num; j++, uptr++) {
				if (UNIT_G_CHAN(uptr->flags) == (unsigned int)chan &&
				    (sms[chan] & 1) ==
				    ((UNIT_SELECT & uptr->flags) != 0)) {
				    goto found;
				}
			    }
			}
			/* If no device, stop right now */
			chan9_set_error(chan, SNS_ADCHECK);
			chan_sense[chan] &=
			    ~(SNS_PREAD | SNS_PWRITE | CTL_SNS1 |
			      CTL_CNTL);
			iotraps |= 1 << chan;
			chan_status[chan] &= ~STA_ACTIVE;
			break;
		      found:
			/* Get channel ready to transfer */
			chan_sense[chan] &= ~CTL_END;
			chan_status[chan] &= ~(DEV_REOR | DEV_FULL);

			/* Call device to start it running */
			switch (dibp->cmd(uptr, cmd[chan], sms[chan])) {
			case 0:
			case -1:
			    chan9_set_error(chan, SNS_IOCHECK);
			    iotraps |= 1 << chan;
			    chan_sense[chan] &= ~(SNS_PREAD | SNS_PWRITE |
						  CTL_SNS1 | CTL_CNTL);
			    chan_status[chan] &= ~STA_ACTIVE;
			    continue;
			case 2:	/* Device not ready yet, wait */
			    continue;
			case 1:	/* Device will be waiting for command */
			    break;
			}
		    }
		    /* Special out for sense command */
		    if (cmd[chan] == SNS) {
			chan_status[chan] &= ~DEV_WRITE;
			chan_status[chan] |= STA_SEL;
			break;
		    }
		    chan_status[chan] |= DEV_WRITE;
		  xfer:
		    /* Check if comand tranfer done */
		    if (chan_status[chan] & DEV_REOR) {
			chan_status[chan] &=
			    ~(DEV_WRITE | DEV_REOR | DEV_FULL);
			chan_sense[chan] &= ~(SNS_READ | SNS_WRITE);
			if ((chan_sense[chan] & CTL_END) == 0)
			    chan_sense[chan] |= (chan_sense[chan] &
						 (SNS_PREAD | SNS_PWRITE))
				>> 2;
			else
			    chan_status[chan] &= ~STA_ACTIVE;
			chan_sense[chan] &= ~(CTL_CNTL | SNS_PREAD |
					      SNS_PWRITE | CTL_END);
			if (chan_sense[chan] & SNS_WRITE)
			    chan_status[chan] |= DEV_WRITE;
			break;
		    }

		    /* Check if device ready for next command word */
		    if ((chan_status[chan] & (DEV_WRITE | DEV_FULL)) ==
			DEV_WRITE) {
			assembly[chan] = M[caddr[chan]];
			if (chan_dev.dctrl & cmask)
			    sim_debug(DEBUG_CMD, &chan_dev,
				      "chan %d cmd > %012llo\n",
				      chan, assembly[chan]);
			if (dualcore)
			    caddr[chan] =
				(0100000 & caddr[chan]) |
				((caddr[chan] + 1) & (MEMSIZE - 1));
			else
			    caddr[chan] =
				(caddr[chan] + 1) & (MEMSIZE - 1);
			chan_status[chan] |= DEV_FULL;
		    }
		    continue;

		case LAR:
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    assembly[chan] = M[caddr[chan]];
		    break;
		case SAR:
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    M[caddr[chan]] = assembly[chan];
		    break;
		case CPYP:
		case CPYP2:
		case CPYP3:
		case CPYP4:
		    if (chan_status[chan] & DEV_REOR) {
			if (sms[chan] & 0100) {
			    chan9_set_error(chan, SNS_UEND);
			    if (chan_status[chan] & STA_SEL)
				chan_status[chan] |=
				    (DEV_DISCO | DEV_WEOR);
			    chan_status[chan] &= ~(STA_SEL);
			}
			if (wcount[chan] != 0)
			    chan_status[chan] &= ~DEV_REOR;
		    }
		case CPYD:
		case CPYDX:

		    /* If device ended, quit transfer */
		    if ((chan_sense[chan] & CTL_END) &&
			(chan_status[chan] & DEV_FULL) == 0) {
			/* Disconnect channel if select still active */
			if (chan_status[chan] & STA_SEL) {
			    chan_status[chan] |= (DEV_DISCO);
			    chan_status[chan] &= ~(STA_SEL);
			}
			chan_status[chan] &= ~DEV_REOR;
			chan_sense[chan] &=
			    ~(CTL_SNS1 | CTL_SNS2 | SNS_READ | SNS_WRITE |
			      CTL_END);
			break;
		    }

		    /* Check if we still have a select signal */
		    if ((chan_status[chan] & STA_SEL) == 0 ||
			(chan_sense[chan] & (SNS_READ | SNS_WRITE |
					     CTL_SNS1 | CTL_SNS2)) == 0) {
			chan9_seqcheck(chan);
			break;
		    }

		    /* Check if last word transfered */
		    if ((chan_status[chan] & DEV_FULL) == 0) {
			if (wcount[chan] == 0) {
			    if (cmd[chan] == CPYD || cmd[chan] == CPYDX ||
				chan_sense[chan] & SNS_UEND) {
				chan_status[chan] |=
				    (DEV_DISCO | DEV_WEOR);
				chan_status[chan] &= ~(STA_SEL);
				chan_sense[chan] &=
				    ~(CTL_SNS1 | CTL_SNS2 | SNS_READ |
				      SNS_WRITE);
			    } else {
				if (chan_status[chan] & DEV_REOR)
				    chan_status[chan] &= ~DEV_REOR;
			    }
			    break;
			    /* Check for record end in non-concurrent IRQ mode */
			} else if (chan_status[chan] & DEV_REOR &&
				   sms[chan] & 0100) {
			    chan9_set_error(chan, SNS_UEND);
			    chan_status[chan] |= (DEV_DISCO | DEV_WEOR);
			    chan_status[chan] &= ~(STA_SEL);
			}
		    }

		    /* Check if ready to transfer something */
		    switch (chan_status[chan] & (DEV_WRITE | DEV_FULL)) {
		    case DEV_WRITE | DEV_FULL:
		    case 0:
			continue;	/* Do nothing if no data xfer */
		    case DEV_WRITE:	/* Device needs data word */
			assembly[chan] = M[caddr[chan]];
			if (chan_dev.dctrl & cmask)
			    sim_debug(DEBUG_DATA, &chan_dev,
				      "chan %d cmd > %012llo\n",
				      chan, assembly[chan]);
			if (sms[chan] & 020)	/* BCD Xlat mode */
			    bcd_xlat(chan, 0);
			if (dualcore) {
			    uint16              t = caddr[chan] & 0100000;

			    if (sms[chan] & 040)	/* Read backward */
				caddr[chan]--;
			    else
				caddr[chan]++;
			    caddr[chan] &= (MEMSIZE - 1);
			    caddr[chan] |= t;
			} else {
			    if (sms[chan] & 040)	/* Read backward */
				caddr[chan]--;
			    else
				caddr[chan]++;
			    caddr[chan] &= MEMSIZE - 1;
			}
			wcount[chan]--;
			chan_status[chan] |= DEV_FULL;
			break;
		    case DEV_FULL:	/* Device has given us a dataword */
			if (sms[chan] & 020)	/* BCD Xlat mode */
			    bcd_xlat(chan, 1);
			if (chan_dev.dctrl & cmask)
			    sim_debug(DEBUG_DATA, &chan_dev,
				      "chan %d cmd < %012llo\n",
				      chan, assembly[chan]);
			M[caddr[chan]] = assembly[chan];
			if (dualcore) {
			    uint16              t = caddr[chan] & 0100000;

			    if (sms[chan] & 040)	/* Read backward */
				caddr[chan]--;
			    else
				caddr[chan]++;
			    caddr[chan] &= (MEMSIZE - 1);
			    caddr[chan] |= t;
			} else {
			    if (sms[chan] & 040)	/* Read backward */
				caddr[chan]--;
			    else
				caddr[chan]++;
			    caddr[chan] &= MEMSIZE - 1;
			}
			wcount[chan]--;
			chan_status[chan] &= ~DEV_FULL;
			break;
		    }

		    /* Tell device this is last word */
		    if (wcount[chan] == 0 &&
			(cmd[chan] == CPYD || cmd[chan] == CPYDX)) {
			chan_status[chan] |= DEV_WEOR;
		    }
		    continue;

		case TCM:
		case TCMX:
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			continue;
		    } else {
			/* Compare wordcount high to wordcound low */
			/* 0 = chan check, 1-6 = assmebly, 7 = 0 */
			uint8               v;
			uint8               ch = wcount[chan] >> 12;
			uint8               mask = wcount[chan] & 077;
			uint8               flag = wcount[chan] & 0100;

			if (ch == 0) {
			    v = (chan_sense[chan] >> 5) & 077;
			} else if (ch == 7) {
			    v = 0;
			} else {
			    v = (uint8)(077 & (assembly[chan] >> (6 * (6 - ch))));
			}
			if ((v == mask && flag == 0)
			    || ((v & mask) == mask && flag != 0))
			    location[chan] = caddr[chan];
		    }
		    break;
		case TDC:
		    if (counter[chan] != 0) {
			location[chan] = caddr[chan];
			counter[chan]--;
		    }
		    break;
		case LCC:
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    counter[chan] = caddr[chan] & 077;
		    break;
		case SMS:
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    sms[chan] = caddr[chan] & 0177;
		    /* Check to see if IRQ still pending */
		    if (chan_sense[chan] & SNS_IRQS &
			(~((sms[chan] << 5) & (SNS_IMSK ^ SNS_IRQS)))) {
			chan_irq[chan] = 1;
		    } else {
			chan_irq[chan] = 0;
		    }
		    break;
		case ICC:
		case ICCX:
		    if (chan_status[chan] & STA_SEL) {
			chan9_seqcheck(chan);
			break;
		    }
		    /* transfer counter from wordcount high to assembly */
		    /* 0 = SMS to 6, 1-6 = assmebly, 7 = nop */
		    {
			t_uint64            v = counter[chan] & 077;
			uint8               ch = wcount[chan] >> 12;

			if (ch == 0) {
			    /* Not what POO says, but what diags want */
			    /* POO says other digits not affected. */
			    assembly[chan] = sms[chan] & 00137;
			} else if (ch != 7) {
			    assembly[chan] &= ~(077L << (6 * (6 - ch)));
			    assembly[chan] |= (v << (6 * (6 - ch)));
			}
		    }
		    break;
		}
	    }

	    /* Check for intrupts */
	    if ((chan_status[chan] & (STA_SEL | STA_ACTIVE)) == 0 &&
		/* Can only interupt when paused on WTR */
		
		(chan_sense[chan] &
		 (CTL_CNTL | CTL_SNS1 | CTL_SNS2 | CTL_INHB)) == 0
		&& (chan_sense[chan] & SNS_IRQS &
		    (~((sms[chan] << 5) & (SNS_IMSK ^ SNS_IRQS))))) {
		M[040 + (chan * 2)] = location[chan] & (MEMSIZE - 1);
		M[040 + (chan * 2)] |= ((t_uint64) caddr[chan]) << 18;
		chan_status[chan] |= STA_ACTIVE;
		chan_sense[chan] |= CTL_INHB;
		location[chan] = 041 + (chan * 2);
		//chan_irq[chan] = 0;
		chan_fetch(chan);
		/* Fake a xec type trap */
		location[chan] = (uint16)M[040 + (chan * 2)] & (MEMSIZE - 1);
		goto again;
	    }

	    if (chan_status[chan] & STA_ACTIVE) {
		chan_fetch(chan);
		if (sms[chan] & 0100 && (cmd[chan] & 071) == CPYD &&
		    wcount[chan] == 0)
		    chan9_set_error(chan, SNS_UEND);
	    }
	}
    }
}

void
chan_fetch(int chan)
{
    uint16              loc = location[chan];
    t_uint64            temp = M[loc & (MEMSIZE - 1)];

    sim_interval--;
    location[chan] = (loc + 1) & (MEMSIZE - 1);
    caddr[chan] = (uint16)temp & (MEMSIZE - 1);
    if (dualcore)
	caddr[chan] |= temp & 0100000;
    wcount[chan] = (uint16)(temp >> 18) & 077777;
    cmd[chan] = (uint8)(((temp >> 30) & 074) | ((temp >> 16) & 1));
    /* Check indirect bit */
    if (temp & 0400000) {
	caddr[chan] = (uint16)M[caddr[chan]];
	if (dualcore)
	    caddr[chan] &= 0100000 | (MEMSIZE - 1);
	else
	    caddr[chan] &= (MEMSIZE - 1);
	sim_interval--;
    }
    if (chan_dev.dctrl & (0x0100 << chan))
	sim_debug(DEBUG_CHAN, &chan_dev,
		  "chan %d fetch adr=%05o cmd=%03o caddr=%05o wcount=%05o\n",
		  chan, location[chan], cmd[chan], caddr[chan],
		  wcount[chan]);
}

/* Sets the device onto a given channel */
t_stat
set_chan(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    DEVICE             *dptr;
    DIB                *dibp;
    int                 newch;
    int                 chan = UNIT_G_CHAN(uptr->flags);

    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    dptr = find_dev_from_unit(uptr);
    if (dptr == NULL)
	return SCPE_IERR;
    dibp = (DIB *) dptr->ctxt;
    if (dibp == NULL)
	return SCPE_IERR;
    if (cptr[1] != '\0')
	return SCPE_ARG;
    if (cptr[0] == '*') {
	newch = 0;
    } else {
	newch = 1 + (*cptr - 'A');
	if (newch < 1 || newch > NUM_CHAN)
	    return SCPE_ARG;
    }
    if (newch == chan)
	return SCPE_OK;
    if ((uptr->flags & UNIT_DIS) == 0) {
	int                 type = CHAN_G_TYPE(chan_unit[chan].flags);
	int                 ctype = dibp->ctype;

	if (ctype < CHAN_7289) {
	    if (chan == 0)
		ctype = CHAN_704;
	    else
		ctype = CHAN_7607;
	}

	if (ctype != type) {
	    if ((chan_unit[chan].flags & CHAN_SET) ||
		((chan_unit[chan].flags & CHAN_MODE)
		 && num_devs[chan] != 0)) {
		return SCPE_IERR;
	    } else {
		chan_unit[chan].flags &= ~CHAN_MODEL;
		chan_unit[chan].flags |= CHAN_S_TYPE(ctype);
	    }
	}
	num_devs[chan]--;
	/* Let channel float if no devices */
	if (num_devs[chan] == 0 && (chan_unit[chan].flags & CHAN_MODE))
	    chan_unit[chan].flags &= ~CHAN_SET;

	num_devs[newch]++;
	chan_unit[newch].flags |= CHAN_SET;
	/* Set the new channel */
	uptr->flags &= ~UNIT_S_CHAN(17);
	uptr->flags |= UNIT_S_CHAN(newch);
    }
    return SCPE_OK;
}

/* Sets the controller onto a given channel */
t_stat
set_cchan(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    DEVICE             *dptr;
    DIB                *dibp;
    int                 newch;
    int                 chan = UNIT_G_CHAN(uptr->flags);
    t_stat              r;
    uint32              unit;
    int                 ctype, type;

    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    dptr = find_dev_from_unit(uptr);
    if (dptr == NULL)
	return SCPE_IERR;
    dibp = (DIB *) dptr->ctxt;
    if (dibp == NULL)
	return SCPE_IERR;
    if (cptr[1] != ' ' || cptr[1] != '\0')
	return SCPE_ARG;
    if (*cptr == '*') {
	newch = 0;
    } else {
	newch = 1 + (*cptr - 'A');
	if (newch < 1 || newch > NUM_CHAN)
	    return SCPE_ARG;
    }
    if (newch == chan)
	return SCPE_OK;
    r = SCPE_OK;

/* First check if we can make change without conflict */
    type = CHAN_G_TYPE(chan_unit[newch].flags);
    ctype = dibp->ctype;
    if (ctype < CHAN_7289) {
	if (chan == 0)
	    ctype = CHAN_704;
	else
	    ctype = CHAN_7607;
    }

    if (ctype != type) {
	if ((chan_unit[newch].flags & CHAN_SET) ||
	    ((chan_unit[newch].flags & CHAN_MODE) && num_devs[newch] != 0)) {
	    return SCPE_IERR;
	} else {
	    chan_unit[newch].flags &= ~CHAN_MODEL;
	    chan_unit[newch].flags |= CHAN_S_TYPE(ctype);
	}
    }

/* All set, switch devices over to new channel */
    num_devs[chan]--;
    if (num_devs[chan] == 0 && (chan_unit[chan].flags & CHAN_MODE))
	chan_unit[chan].flags &= ~CHAN_SET;
    chan_unit[newch].flags |= CHAN_SET;
    num_devs[newch]++;
    for (unit = 0; unit < dptr->numunits; unit++) {
	/* Set the new channel */
	uptr[unit].flags &= ~UNIT_S_CHAN(17);
	uptr[unit].flags |= UNIT_S_CHAN(newch);
    }
    return SCPE_OK;
}

t_stat
get_chan(FILE * st, UNIT * uptr, int32 v, void *desc)
{
    DEVICE             *dptr;
    DIB                *dibp;
    int                 chan = UNIT_G_CHAN(uptr->flags);

    if (uptr == NULL)
	return SCPE_IERR;
    dptr = find_dev_from_unit(uptr);
    if (dptr == NULL)
	return SCPE_IERR;
    dibp = (DIB *) dptr->ctxt;
    if (dibp == NULL)
	return SCPE_IERR;
    if (chan == 0)
	fprintf(st, "704 device");
    else
	fprintf(st, "Chan=%c", chan + 'A' - 1);
    return SCPE_OK;
}

t_stat
chan9_set_select(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_ATT)
	return SCPE_ALATT;
    if (*cptr == '\0' || cptr[1] != '\0')
	return SCPE_ARG;
    if (*cptr == '0')
	uptr->flags &= ~UNIT_SELECT;
    else if (*cptr == '1')
	uptr->flags |= UNIT_SELECT;
    else
	return SCPE_ARG;
    return SCPE_OK;
}

t_stat
chan9_get_select(FILE * st, UNIT * uptr, int32 v, void *desc)
{
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_SELECT)
	fputs("Select=1", st);
    else
	fputs("Select=0", st);
    return SCPE_OK;
}

/* Reset the channel, clear any pending device */
void
chan_rst(int chan, int type)
{
    /* Issure reset command to device */
    if (type && CHAN_G_TYPE(chan_unit[chan].flags) != CHAN_7909)
	return;
    if (type == 0 && CHAN_G_TYPE(chan_unit[chan].flags) == CHAN_7909)
	return;
    /* Clear outstanding traps on reset */
    if (type)
	iotraps &= ~(1 << chan);
    chan_status[chan] &= ~(STA_ACTIVE | STA_PEND | CHS_ATTN | STA_SEL |
			   STA_WAIT | STA_START);
    chan_info[chan] &= ~(CHAINF_START | CHAINF_RUN);
    chan_sense[chan] = 0;
    caddr[chan] = 0;
    cmd[chan] = 0;
    sms[chan] = 0;
    chan_irq[chan] = 0;
    wcount[chan] = 0;
    location[chan] = 0;
    counter[chan] = 0;		/* Channel memory address */
}

/* Issue a command to a channel */
int
chan_cmd(uint16 dev, uint16 dcmd)
{
    UNIT               *uptr;
    uint32              chan;
    DEVICE            **dptr;
    DIB                *dibp;
    int                 j;

    /* Find device on given channel and give it the command */
    chan = (dev >> 9) & 017;
    /* If no channel device, quick exit */
    if (chan_unit[chan].flags & UNIT_DIS)
	return 0;
    /* On 704 device new command aborts current operation */
    if (CHAN_G_TYPE(chan_unit[chan].flags) == CHAN_704 &&
	(chan_status[chan] & (STA_SEL | DEV_FULL | DEV_DISCO)) == STA_SEL) {
	chan_status[chan] |= DEV_DISCO | DEV_WEOR;
	return 2;
    }
    /* Unit is busy doing something, wait */
    if (chan_status[chan] & (STA_SEL | DEV_DISCO))
	return 2;
    /* Ok, try and find the unit */
    dev &= 0777;
    for (dptr = sim_devices; *dptr != NULL; dptr++) {
	int                 r;

	dibp = (DIB *) (*dptr)->ctxt;
	/* If no DIB, not channel device */
	if (dibp == NULL || dibp->ctype == CHAN_7909 ||
	    (dibp->addr & dibp->mask) != (dev & dibp->mask))
	    continue;
	uptr = (*dptr)->units;
	if (dibp->upc == 1) {
	    int                 num = (*dptr)->numunits;

	    for (j = 0; j < num; j++) {
		if (UNIT_G_CHAN(uptr->flags) == chan) {
		    r = dibp->cmd(uptr, dcmd, dev);
		    if (r != -1)
			return r;
		}
		uptr++;
	    }
	} else {
	    if (UNIT_G_CHAN(uptr->flags) == chan) {
		r = dibp->cmd(uptr, dcmd, dev);
		if (r != -1)
		    return r;
	    }
	}
    }
    /* Check if this is a IOD instruction */
    /* If we have gotten here, means that everything is stoped */
    if (dcmd == OP_WRS && chan == 0 && dev == 0333) {
	return 1;
    }
    return -1;
}

/* Check channel for error */
int chan_error(int chan)
{
    return chan_status[chan] & CHS_ATTN;
}

/* Check channel at eof */
int chan_stat(int chan, uint16 flag)
{
    if (chan_status[chan] & flag) {
	chan_status[chan] ^= flag;
	return 1;
    }
    return 0;
}

/* Check channel is selected */
int chan_select(int chan)
{
    return chan_status[chan] & STA_SEL;
}

/* Check channel is active */
int chan_active(int chan)
{
    return (chan_status[chan] & 
		(STA_SEL | STA_ACTIVE | STA_WAIT | STA_TWAIT)) != 0;
}

/* Give channel a new address to start working at */
int
chan_start(int chan, uint16 addr)
{
    /* Hold this command until after channel has disconnected */
    if (chan_status[chan] & DEV_DISCO)
	return 1;

    /* Depending on channel type controls how command works */
    if (CHAN_G_TYPE(chan_unit[chan].flags) == CHAN_7909) {
	if (chan_status[chan] & STA_ACTIVE)
	    return 1;
	chan_sense[chan] &=
	    ~(CTL_CNTL | CTL_SNS1 | CTL_SNS2 | SNS_READ | SNS_PREAD |
	      SNS_WRITE | SNS_PWRITE);
    } else {
	/* All clear, start ball rolling on new command */
	/* Force iocheck if attempt to load inactive channel with command */
	if ((chan_status[chan] & STA_SEL) == 0)
	    return -1;
    }
    if (chan_dev.dctrl & (0x0100 << chan))
	sim_debug(DEBUG_CMD, &chan_dev,
		  "chan %d start IC=%05o addr=%o\n\r", chan, IC - 1, addr);
    /* Clear pending traps for this channel */
    iotraps &= ~(1 << chan);
    /* Fetch next command */
    location[chan] = addr;
    chan_fetch(chan);
    chan_status[chan] |= STA_START | STA_ACTIVE;
    chan_info[chan] |= CHAINF_START;
    return 0;
}

/* Give channel a new address to start working at */
int
chan_load(int chan, uint16 addr)
{
    if (CHAN_G_TYPE(chan_unit[chan].flags) == CHAN_7909) {
	if (chan_status[chan] & STA_ACTIVE)
	    return 1;
	if (cmd[chan] == TWT) {
	    /* Ignore command if this waiting on IRQ */
	    if (iotraps & (1 << chan))
		return 0;
	    chan_sense[chan] &= ~CTL_INHB;
	}
	location[chan] = caddr[chan];
    } else {
	/* Force iocheck if attempt to load channel with command */
	if ((chan_status[chan] & STA_SEL) == 0)
	    return -1;
	/* If channel active, pend command, this should hold CPU */
	if (chan_status[chan] & (STA_ACTIVE | STA_WAIT)) {
	    chan_status[chan] |= STA_PEND;
	    return 1;
	}
	/* Can't do load if channel has not been started */
	if ((chan_status[chan] & STA_START) == 0)
	    return -1;
	chan_status[chan] &= ~STA_PEND;
	location[chan] = addr;
    }
    /* All clear, start ball rolling on new command */
    if (chan_dev.dctrl & (0x0100 << chan))
	sim_debug(DEBUG_CMD, &chan_dev,
		  "chan %d load IC=%05o addr=%o\n\r", chan, IC - 1, addr);
    chan_fetch(chan);
    chan_status[chan] |= STA_ACTIVE;
    return 0;
}

/* return the channels current command address */
void
chan_store(int chan, uint16 loc)
{
    t_uint64            reg = 0L;

    /* Check if channel has units attached */
    if (chan_unit[chan].flags & CHAN_SET) {
	/* Return command/addr/xcmd/location */
	if (CHAN_G_TYPE(chan_unit[chan].flags) == CHAN_7909) {
	    reg = location[chan] & (MEMSIZE - 1);
	    reg |= ((t_uint64) caddr[chan]) << 18;
	    /* Check if channel has units attached */
	} else {
	    reg = caddr[chan];
	    reg |= ((t_uint64) (location[chan] & (MEMSIZE - 1))) << 18;
	    reg |= ((t_uint64) (cmd[chan] & 070)) << 30;
	    reg |= ((t_uint64) (cmd[chan] & 01)) << 16;
	}
    }
    M[loc & (MEMSIZE - 1)] = reg;
}

/* Nop for the momement */
void
chan_store_diag(int chan, uint16 loc)
{
    t_uint64            reg;
    int                 results;

    /* Counter[6], iocheck,seq check, unusal end, attn 1,
     * attn 2, adpter check, prepare to read, prepare to write,
     * read status, write status, interupt */
    if (CHAN_G_TYPE(chan_unit[chan].flags) == CHAN_7909) {
	reg = ((t_uint64) counter[chan]) << 30;
	results = SNS_MASK & chan_sense[chan]	/* &
						 * (~((sms[chan] << 5) & (SNS_IMSK ^ SNS_IRQS))) */ ;
	reg |= ((t_uint64) (results)) << 19;
	if (chan_irq[chan])
	    reg |= 1 << 19;
	M[loc & (MEMSIZE - 1)] = reg;
    }
}

/*
 * Write a word to the assembly register.
 */
int
chan_write(int chan, t_uint64 * data, int flags)
{

    /* Check if any data available yet */
    if (chan_status[chan] & DEV_FULL) {
	/* Nope, see if we are waiting for end of record. */
	if (chan_status[chan] & DEV_WEOR) {
	    chan_status[chan] &= ~DEV_WEOR;
	    chan_status[chan] |= DEV_REOR;
	    return END_RECORD;
	}
	if (chan_status[chan] & STA_ACTIVE) {
	    chan_status[chan] |= CHS_ATTN;	/* We had error */
	    if ((flags & DEV_DISCO) == 0)
		iocheck = 1;
	}
	if (chan == 0) {
	    chan_status[chan] &= ~STA_SEL;
	    chan_status[chan] |= DEV_DISCO;
	}
	return TIME_ERROR;
    } else {
	assembly[chan] = *data;
	chan_status[chan] |= DEV_FULL;
	chan_status[chan] &= ~DEV_WRITE;
	if (flags & DEV_REOR) {
	    chan_status[chan] |= DEV_REOR;
	}
    }

    /* If Writing end of record, abort */
    if (flags & DEV_WEOR) {
	chan_status[chan] &= ~(DEV_FULL | DEV_WEOR);
	return END_RECORD;
    }

    return DATA_OK;
}

/*
 * Read next word from assembly register.
 */
int
chan_read(int chan, t_uint64 * data, int flags)
{

    /* Return END_RECORD if requested */
    if (flags & DEV_WEOR) {
	chan_status[chan] &= ~(DEV_WEOR | STA_WAIT);
	return END_RECORD;
    }
    /* Check if he write out last data */
    if ((chan_status[chan] & DEV_FULL) == 0) {
	if (chan_status[chan] & DEV_WEOR) {
	    if ((chan_status[chan] & (STA_WAIT | STA_PEND)) !=
		(STA_WAIT | STA_PEND)) chan_status[chan] |= STA_ACTIVE;
	    chan_status[chan] |= DEV_WRITE;
	    chan_status[chan] &= ~(DEV_WEOR | STA_WAIT);
	    return END_RECORD;
	}
	if (chan_status[chan] & STA_ACTIVE) {
	    chan_status[chan] |= CHS_ATTN;
	    if ((flags & DEV_DISCO) == 0)
		iocheck = 1;
	}
	if (chan == 0) {
	    chan_status[chan] &= ~STA_SEL;
	    chan_status[chan] |= DEV_DISCO;
	}
	return TIME_ERROR;
    } else {
	*data = assembly[chan];
	chan_status[chan] &= ~DEV_FULL;
	/* If end of record, don't transfer any data */
	if (flags & DEV_REOR) {
	    chan_status[chan] &= ~(DEV_WRITE | STA_WAIT);
	    chan_status[chan] |= DEV_REOR;
	} else
	    chan_status[chan] |= DEV_WRITE;
    }
    return DATA_OK;
}

void
chan_set_attn(int chan)
{
    chan_status[chan] |= CHS_ATTN;
}

void
chan_set_eof(int chan)
{
    chan_status[chan] |= CHS_EOF;
}

void
chan_set_error(int chan)
{
    chan_status[chan] |= CHS_ERR;
}

void
chan_set_sel(int chan, int need)
{
    chan_status[chan] &=
	~(DEV_WEOR | DEV_REOR | DEV_FULL | DEV_WRITE | DEV_DISCO);
    chan_status[chan] |= STA_SEL;
    if (need)
	chan_status[chan] |= DEV_WRITE;
}

void
chan_clear_status(int chan)
{
    chan_status[chan] &=
	~(CHS_ATTN | CHS_EOT | CHS_BOT | CHS_EOF | DEV_REOR | DEV_WEOR);
}

void
chan_set(int chan, uint16 flag)
{
    chan_status[chan] |= flag;
}

void
chan_clear(int chan, uint16 flag)
{
    chan_status[chan] &= ~flag;
}

void
chan9_seqcheck(int chan)
{
    /* Disconnect channel if active */
    if (chan_status[chan] & STA_SEL)
	chan_status[chan] |= DEV_DISCO;
    chan_sense[chan] &= ~(SNS_READ | SNS_WRITE | CTL_SNS1 | CTL_SNS2);
    chan_status[chan] &= ~(STA_SEL | STA_ACTIVE);
    chan9_set_error(chan, SNS_SEQCHECK);
}

void
chan9_set_attn(int chan, int sel)
{
    uint16              mask = SNS_ATTN1 << sel;

    chan9_set_error(chan, mask);
}

void
chan9_set_error(int chan, uint16 mask)
{
    if (chan_sense[chan] & mask)
	return;
    chan_sense[chan] |= mask;
    if (mask & (~((sms[chan] << 5) & (SNS_IMSK ^ SNS_IRQS)))) {
	chan_irq[chan] = 1;
    }
}
