/* i7090_chron.c: IBM 7090 Chrono clock on MT drive.

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


*/

#include "i7090_defs.h"
#include <time.h>

#define NUM_DEVS_CHRON	1
#define BUFFSIZE	(3)

#define UNIT_MT(x)	UNIT_DISABLE | UNIT_ROABLE | \
			UNIT_S_CHAN(x)

/* in u3 is device address */
/* in u4 is current buffer position */
/* in u5 */
#define MT_RDS		1
#define MT_RDSB		2
#define MT_SKIP		11	/* Do skip to end of record */
#define MT_CMDMSK   000017	/* Command being run */
#define MT_RDY 	    000020	/* Device is ready for command */
#define MT_IDLE     000040	/* Tape still in motion */
#define MT_EOR      000200	/* Hit end of record */
#define MT_ERR	    000400	/* Device recieved error */
#define MT_BOT	    001000	/* Unit at begining of tape */
#define MT_EOT	    002000	/* Unit at end of tape */

uint32              chron(UNIT *, uint16, uint16);
t_stat              chron_srv(UNIT *);
t_stat              chron_reset(DEVICE *);
t_stat              set_addr(UNIT * uptr, int32 val, char *cptr,

			     void *desc);
t_stat              get_addr(FILE * st, UNIT * uptr, int32 v, void *desc);

/* One buffer per channel */
t_uint64            chron_buffer[BUFFSIZE];
DIB                 chron_dib = { CHAN_7607, 1, 0201, 0740, &chron, NULL };

UNIT                chron_unit[] = {
/* Controller 1 */
    {UDATA(&chron_srv, UNIT_MT(1) | UNIT_DIS, 0), 10},	/* 0 */
};

MTAB                chron_mod[] = {
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "UNIT", "UNIT", &set_addr, &get_addr,
     NULL},
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "CHAN", "CHAN", &set_chan, &get_chan,
     NULL},
    {0}
};

DEVICE              chron_dev = {
    "CHRON", chron_unit, NULL, chron_mod,
    NUM_DEVS_CHRON, 8, 15, 1, 8, 36,
    NULL, NULL, &chron_reset, NULL, NULL, NULL,
    &chron_dib, DEV_DISABLE
};

uint32 chron(UNIT * uptr, uint16 cmd, uint16 dev)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    int                 time = 30;
    int                 unit = (dev & 017) - 1;

    /* Make sure valid drive number */
    if (unit != uptr->u3)
	return -1;
    if (uptr->flags & UNIT_DIS)
	return -1;

    /* Check if drive is ready to recieve a command */
    if ((uptr->u5 & MT_RDY) == 0) {
	/* Return indication if not ready and doing TRS */
	if (cmd == OP_TRS) {
	    return 0;
	} else
	    return 2;
    }
    uptr->u5 &= ~(MT_CMDMSK | MT_RDY);
    switch (cmd) {
    case OP_RDS:
	if (dev & 020)
	    uptr->u5 |= MT_RDSB;
	else
	    uptr->u5 |= MT_RDS;
	time = 100;
	chan_set_sel(chan, 0);
	chan_clear_status(chan);
	uptr->u6 = 0;
	break;
    case OP_WRS:
	/* Can't write to it so return error */
	return -1;

    case OP_BSR:
	/* Nop, just set us back at begining */
	chan_set(chan, CHS_BOT);
	return 1;
    case OP_BSF:
	/* Nop, just set flag and leave */
	chan_set(chan, CHS_BOT);
	return 1;
	/* All nops, just return success */
    case OP_WEF:
    case OP_REW:
    case OP_RUN:
    case OP_SDN:
    case OP_TRS:
	return 1;
    }
    sim_cancel(uptr);
    sim_activate(uptr, time);
    return 1;
}

/* Chronolog clock */

/* Convert number (0-99) to BCD */

static void
bcd_2d(int n, uint8 * b2)
{
    uint8               d1, d2;

    d1 = n / 10;
    d2 = n % 10;
    *b2++ = d1;
    *b2 = d2;
}

void
chron_read_buff(UNIT * uptr, int cmd)
{
    time_t              curtim;
    struct tm          *tptr;
    uint8               buffer[12];
    int                 words;
    t_uint64            wd = 0;
    int                 i, j;
    int                 ms;

    uptr->u6 = 0;		/* Set to no data */

    curtim = time(NULL);	/* get time */
    tptr = localtime(&curtim);	/* decompose */
    if (tptr == NULL)
	return;			/* error? */
    ms = sim_os_msec() % 1000;
    ms /= 60;

    bcd_2d(tptr->tm_mon + 1, &buffer[0]);
    bcd_2d(tptr->tm_mday, &buffer[2]);
    bcd_2d(tptr->tm_hour, &buffer[4]);
    bcd_2d(tptr->tm_min, &buffer[6]);
    bcd_2d(tptr->tm_sec,  &buffer[8]);
    bcd_2d(ms,  &buffer[10]);

    uptr->u6 = words = BUFFSIZE - 1;
    /* Copy data to buffer */
    for (j = 6, i = 0; i < 12; i++) {
	wd |= ((t_uint64) buffer[i]) << (6 * --j);
	if (j == 0) {
	    chron_buffer[words--] = wd;
	    wd = 0;
	    j = 6;
	}
    }
    return;
}

t_stat chron_srv(UNIT * uptr)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    int                 cmd = uptr->u5 & MT_CMDMSK;

    /* Channel has disconnected, abort current read. */
    if (chan_status[chan] & DEV_DISCO && (uptr->u5 & MT_RDY) == 0) {
	uptr->u5 &= ~MT_CMDMSK;
	if (cmd = MT_RDS || cmd == MT_RDSB) {
	    uptr->u6 = 0;
	    chan_clear(chan, STA_SEL);
	}
	uptr->u5 |= MT_RDY;
	chan_clear(chan, DEV_DISCO | DEV_WEOR);
	return SCPE_OK;
    }

    switch (uptr->u5 & MT_CMDMSK) {
    case 0:			/* No command, stop tape */
	uptr->u5 |= MT_RDY;	/* Ready since command is done */
	break;

    case MT_SKIP:		/* Record skip done, enable tape drive */
	sim_activate(uptr, 500);
	break;

    case MT_RDS:
    case MT_RDSB:
	if (uptr->u6 == 0)
	    chron_read_buff(uptr, cmd);
	switch (chan_write(chan, &chron_buffer[uptr->u6],
			   (uptr->u6 == 1) ? DEV_REOR : 0)) {
	case DATA_OK:
	    uptr->u6--;
	    sim_activate(uptr, 50);
	    break;

	case END_RECORD:
	case TIME_ERROR:
	    uptr->u5 &= ~MT_CMDMSK;
	    uptr->u5 |= MT_SKIP;
	    sim_activate(uptr, 50);
	    uptr->u6 = 0;	/* Force read next record */
	    break;
	}

    }
    return SCPE_OK;
}

t_stat
chron_reset(DEVICE * dptr)
{
    chron_unit[0].u5 = MT_RDY;
    return SCPE_OK;
}

/* Sets the address of the chrono clock */
t_stat
set_addr(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    int                 i;

    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_ATT)
	return SCPE_ALATT;
    i = 0;
    while (*cptr != '\0') {
	if (*cptr < '0' || *cptr > '9')
	    return SCPE_ARG;
	i = (i * 10) + (*cptr++) - '0';
    }
    if (i < 0 || i > 10)
	return SCPE_ARG;
    uptr->u3 = i;
    return SCPE_OK;
}

t_stat
get_addr(FILE * st, UNIT * uptr, int32 v, void *desc)
{
    if (uptr == NULL)
	return SCPE_IERR;
    fprintf(st, "Unit=%d", uptr->u3);
    return SCPE_OK;
}
