/* i7090_drum.c: IBM 7090 Drum

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   This supports the direct channel and 704 type drums.

*/

#include "i7090_defs.h"

#define NUM_UNITS_DR	16
#define NUM_DEVS_DR	1
#define UNIT_DRM	UNIT_ATTABLE | UNIT_DISABLE | UNIT_FIX | \
			UNIT_BUFABLE | UNIT_MUSTBUF

/* Device status information stored in u5 */
#define DRMSTA_READ	000001	/* Unit is in read */
#define DRMSTA_WRITE	000002	/* Unit is in write */
#define DRMSTA_CMD	000004	/* Unit has recieved a cmd */
#define DRMSTA_UNIT	000170	/* Unit mask */
#define DRMSTA_UNITSHIFT     3
#define DRMSTA_START	000200	/* Drum has started to transfer */
#define DRMWORDTIME	    20	/* Number of cycles per drum word */
#define DRMSIZE		 2048	/* Number words per drum */
#define DRMMASK	     (DRMSIZE-1)/* Mask of drum address */

uint32              drm(UNIT *, uint16, uint16);
t_stat              drm_srv(UNIT *);
t_stat              drm_boot(int32, DEVICE *);
void                drm_ini(UNIT *, t_bool);
t_stat              drm_reset(DEVICE *);
extern t_stat       chan_boot(int32, DEVICE *);
uint32              drum_addr;	/* Read/write drum address */
t_stat              set_units(UNIT * uptr, int32 val, char *cptr,

			      void *desc);
t_stat              get_units(FILE * st, UNIT * uptr, int32 v, void *desc);

DIB                 drm_dib = { CHAN_704, 1, 0301, 0740, &drm, &drm_ini };

UNIT                drm_unit[] = {
    
	{UDATA(&drm_srv, UNIT_S_CHAN(0) | UNIT_DRM, NUM_UNITS_DR * DRMSIZE), 0,
     NUM_UNITS_DR},
};

MTAB                drm_mod[] = {
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "UNITS", "UNITS", &set_units,
     &get_units, NULL},
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "CHAN", "CHAN", &set_chan, &get_chan,
     NULL},
    {0}
};

DEVICE              drm_dev = {
    "DR", drm_unit, NULL /* Registers */ , drm_mod,
    NUM_DEVS_DR, 8, 15, 1, 8, 36,
    NULL, NULL, &drm_reset, &drm_boot, NULL, NULL,
    &drm_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};

uint32 drm(UNIT * uptr, uint16 cmd, uint16 dev)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    int                 u = (dev - 1) & 037;

    if (u > uptr->u3)
	return -1;
    if ((uptr->flags & UNIT_ATT) != 0) {
	/* Wait for device to time out */
	if (uptr->u5 & DRMSTA_CMD)
	    return 2;
	switch (cmd) {
	case OP_RDS:
	    /* Start device */
	    uptr->u5 = DRMSTA_READ | DRMSTA_CMD;
	    sim_debug(DEBUG_CMD, &drm_dev, "RDS %o\n", dev);
	    chan_set_sel(chan, 0);
	    break;
	case OP_WRS:
	    /* Start device */
	    uptr->u5 = DRMSTA_WRITE | DRMSTA_CMD;
	    sim_debug(DEBUG_CMD, &drm_dev, "WRS %o\n", dev);
	    chan_set_sel(chan, 1);
	    break;
	default:
	    return 0;
	}
	/* Choose which part to use */
	uptr->u5 |= u << DRMSTA_UNITSHIFT;
	drum_addr = 0;		/* Set drum address */
	return 1;
    }
    return 0;
}

t_stat drm_srv(UNIT * uptr)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    t_uint64           *buf = uptr->filebuf;
    t_stat              r;

    uptr->u6++;			/* Adjust rotation */
    uptr->u6 &= DRMMASK;
    /* Channel has disconnected, abort current read. */
    if (chan_status[chan] & DEV_DISCO && uptr->u5 & DRMSTA_CMD) {
	uptr->u5 = 0;
	chan_clear(chan, DEV_DISCO | DEV_WEOR | STA_SEL);
	sim_debug(DEBUG_CHAN, &drm_dev, "Disconnect\n");
    }

    /* Check if we have a address match */
    if ((chan_status[chan] & (STA_ACTIVE | STA_SEL)) == (STA_ACTIVE | STA_SEL)
	 && (uptr->u5 & (DRMSTA_READ | DRMSTA_WRITE))
	 && (uint32)uptr->u6 == (drum_addr & DRMMASK)) {
	uint32            addr =
	    (((uptr->u5 & DRMSTA_UNIT) >> DRMSTA_UNITSHIFT) << 11)
	    + (drum_addr & DRMMASK);

	/* Try and transfer a word of data */
	if (uptr->u5 & DRMSTA_READ) {
	    r = chan_write(chan, &buf[addr], DEV_DISCO);
	} else {
	    if (addr >= uptr->hwmark)
	    	uptr->hwmark = (uint32)addr + 1;
	    r = chan_read(chan, &buf[addr], DEV_DISCO);
	}
	switch (r) {
	case DATA_OK:
	    sim_debug(DEBUG_DATA, &drm_dev, "data %012llo\n", buf[addr]);
	    addr++;
	    addr &= DRMMASK;
	    drum_addr &= ~DRMMASK;
	    drum_addr |= addr;
	    break;

	case END_RECORD:
	case TIME_ERROR:
	   /* If no data, disconnect */
	    uptr->u5 = DRMSTA_CMD;
	    chan_clear(chan, STA_SEL);
	    break;
	}
    }
    sim_activate(uptr, DRMWORDTIME);
    return SCPE_OK;
}

/* Boot from given device */
t_stat
drm_boot(int32 unit_num, DEVICE * dptr)
{
    UNIT               *uptr = &dptr->units[unit_num];
    t_uint64           *buf = uptr->filebuf;
    int                 addr;

    if ((uptr->flags & UNIT_ATT) == 0)
	return SCPE_UNATT;	/* attached? */
/* Init for a read */
    if (drm(uptr, OP_RDS, 0301) != 1)
	return STOP_IONRDY;
/* Copy first three records. */
    addr = 0;
    M[0] = buf[addr++];
    M[1] = buf[addr++];
    drum_addr = 2;
    return chan_boot(unit_num, dptr);
}

void
drm_ini(UNIT * uptr, t_bool f)
{
    uptr->u5 = 0;
    sim_activate(uptr, DRMWORDTIME);
}

t_stat
drm_reset(DEVICE * dptr)
{
    return SCPE_OK;
}

/* Sets the number of drum units */
t_stat
set_units(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    int                 i;

    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_ATT)
	return SCPE_ALATT;
    i = 0;
    while (*cptr != '\0') {
	if (*cptr < '0' || *cptr > '9')
	    return SCPE_ARG;
	i = (i * 10) + (*cptr++) - '0';
    }
    if (i < 0 || i > NUM_UNITS_DR)
	return SCPE_ARG;
    uptr->capac = i * 2048;
    uptr->u3 = i;
    return SCPE_OK;
}

t_stat
get_units(FILE * st, UNIT * uptr, int32 v, void *desc)
{
    if (uptr == NULL)
	return SCPE_IERR;
    fprintf(st, "Units=%d", uptr->u3);
    return SCPE_OK;
}
