/* i7090_disk.c: IBM 7090 Disk

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   Support for 1301/1302/2302 disks and 7238 drums

   Disks are represented in files as follows:

   Since these drives supported variable format for each cylinder the
   format is represented as one track per cylinder as follows:
  
     0          data
     1          header
     2          Home Address
     3          end of track

   These codes are packed 4 per byte and used to control read/write of
   data.

   After one format track per cylinder there is one record of bytes per
   track data for each track. First bytes are home address 2, followed
   by record address, and record data to cover number in format. All data
   is stored with the top 2 bits as zero.

   Limitiation of this are that the address field for each record can not
   be more then 16 bytes.

*/

#include "i7090_defs.h"

#define NUM_DEVS_DSK	10
#define UNIT_DSK	UNIT_ATTABLE | UNIT_DISABLE | UNIT_FIX
#define FORMAT_OK	(1 << UNIT_V_LOCAL)
#define HA2_OK		(2 << UNIT_V_LOCAL)

/* Device status information stored in u5 */
#define DSKSTA_CMD	0x0000100	/* Unit has recieved a cmd */
#define DSKSTA_EXEC	0x0000200	/* Unit is executing a cmd */
#define DSKSTA_WRITE	0x0000400	/* Last command was a write */
#define DSKSTA_CHECK	0x0000800	/* Doing a write check */
#define DSKSTA_CMSK	0x00000ff	/* Command mask */
#define DSKSTA_ARGMSK	0x0fff000	/* Command argument */
#define DSKSTA_ARGSHFT  12
#define DSKSTA_SCAN	0x1000000	/* Scanning for header */
#define DSKSTA_SKIP	0x2000000	/* Skiping record */
#define DSKSTA_XFER	0x4000000	/* Tranfser current record */
#define DSKSTA_DIRTY	0x8000000	/* Buffer needs to be written */

#define FMT_DATA	0		/* Data */
#define FMT_HDR		1		/* Header */
#define FMT_HA2		2		/* Home address 2 */
#define FMT_END		3		/* End of track */

/* Disk commands */
#define DNOP		0x00		/* Nop */
#define DREL		0x04		/* Release */
#define DEBM		0x08		/* Eight Bit mode */
#define DSBM		0x09		/* Six bit mode */
#define DSEK		0x80		/* Seek */
#define DVSR		0x82		/* Prepare to Verify single record */
#define DWRF		0x83		/* Prepare to Format */
#define DVTN		0x84		/* Prepare to Verify track no addr */
#define DVCY		0x85		/* Prepare to Verify Cyl */
#define DWRC		0x86		/* Prepare to Write Check */
#define DSAI		0x87		/* Set Access Inoperative */
#define DVTA		0x88		/* Prepare to Verify track addr */
#define DVHA		0x89		/* Prepare to Verify home addr */

/* Disk sense codes */
#define STAT_SIXBIT	00000000000400L	/* Disk in 6bit more. */
#define EXPT_FILECHK	00010000010000L	/* File control check error */
#define EXPT_DSKCHK     00010000020000L	/* Disk storage error */
#define STAT_NOTRDY	00010000040000L	/* Disk no ready */
#define STAT_OFFLINE	00010000200000L	/* Disk offline */
#define DATA_PARITY	00020001000000L	/* Data parity error */
#define DATA_CHECK	00020002000000L	/* Compare error */
#define DATA_RESPONSE   00020004000000L	/* Response check */
#define PROG_INVADDR    00040020000000L	/* Invalid seek address */
#define PROG_NOREC      00040100000000L	/* No record found */
#define PROG_FMTCHK 	00040200000000L	/* Format check */
#define PROG_INVCODE	00040400000000L	/* Invalid code */
#define PROG_INVSEQ	00042000000000L	/* Invalid sequence */

#define MAXTRACK	6020	/* Max size per track */

uint32              dsk(UNIT *, uint16, uint16);
t_stat              dsk_srv(UNIT *);
t_stat              dsk_boot(int32, DEVICE *);
void                dsk_ini(UNIT *, t_bool);
t_stat              dsk_reset(DEVICE *);
t_stat              dsk_set_module(UNIT * uptr, int32 val, char *cptr,
				   void *desc);
t_stat              dsk_get_module(FILE * st, UNIT * uptr, int32 v,
				   void *desc);
t_stat              dsk_set_type(UNIT * uptr, int32 val, char *cptr,
				 void *desc);
t_stat              dsk_get_type(FILE * st, UNIT * uptr, int32 v,
				 void *desc);
int                 disk_rblock(UNIT * uptr, int track);
int                 disk_wblock(UNIT * uptr);
void                disk_posterr(UNIT * uptr, t_uint64 error);
void                disk_cmderr(UNIT * uptr, t_uint64 error);
void                disk_cmd(UNIT * uptr);
int                 disk_write(UNIT * uptr, uint8 data, int chan,
			       int eor);
int                 disk_read(UNIT * uptr, uint8 * data, int chan);
int                 disk_format(UNIT * uptr, FILE * f, int cyl,
				UNIT * base);
int		     bcd_to_track(uint32 addr);
extern uint16       chan_sense[NUM_CHAN];
extern uint16       sms[NUM_CHAN];
extern uint16       IC;

/* Data buffer for track */
uint8               dbuffer[NUM_DEVS_DSK * 4][MAXTRACK];

/* Format buffer for cylinder */
uint8               fbuffer[NUM_DEVS_DSK * 4][MAXTRACK / 4];

/* Currently loaded format record */
uint16              fmt_cyl[NUM_DEVS_DSK * 4];

/* Currently read in track in buffer */
uint16              dtrack[NUM_DEVS_DSK * 4];

/* Arm position */
uint16              arm_cyl[NUM_DEVS_DSK * 4];
uint16              seek_dst[NUM_DEVS_DSK * 4];
uint16              seek_addr[NUM_DEVS_DSK * 4];
t_uint64            sense[NUM_CHAN * 2];
t_uint64            sense_unit[NUM_CHAN * 2];
uint32              cmd_buffer[NUM_CHAN];	/* Command buffer per channel */

/* Macro to help build the disk size table */
#define DISK_DEF(name, cyl, cylpertrk, acc, charpertrk, overhd, mod) \
	{ name, cyl, cylpertrk, ((charpertrk/128) + 1) * 128, \
	 acc, (((charpertrk/128) + 1) * 128)/4, \
	  acc * cyl * ((((charpertrk/128) + 1) * 128)/4), \
	overhd, mod }

struct disk_t
{
    char               *name;	/* Type Name */
    int                 cyl;	/* Number of cylinders */
    int                 track;	/* Number of tracks/cylinder */
    unsigned int        bpt;	/* Max bytes per track */
    int                 arms;	/* Number of access arms */
    int                 fbpt;	/* Number of format bytes per track */
    int                 fmtsz;	/* Format size */
    int                 overhd;	/* Number of characters overhead on HA/RA */
    int                 mods;	/* Number of modules */
}
disk_type[] =
{
    	DISK_DEF("1301", 254, 40, 1, 2880, 4, 1),
	DISK_DEF("1301-2", 254, 40, 1, 2880, 4, 2),
	DISK_DEF("1302", 254, 40, 2, 5940, 7, 1),
	DISK_DEF("1302-2", 254, 40, 2, 5940, 7, 2),
	DISK_DEF("2302", 254, 40, 2, 5940, 7, 2),
	DISK_DEF("7238", 1, 404, 1, 3270, 4, 1),
	DISK_DEF("7238-2", 1, 404, 1, 3270, 4, 2), {
    	NULL, 0}
};

int                 unit_bit[] = {
   4, 2, 1, 0, -1, -3, -4, -5, -7, -9, -36, -36, -36, -36, -36, -36,
  -10, -11, -12, -15, -16, -17, -19, -21, -22, -23, -36, -36, -36, -36, 
		-36, -36,
};

#define DSKSTA_BSY	0x02	/* Controller busy. */
#define DSKSTA_EIGHT	0x04	/* Controller in 8 bit mode */

DIB                 dsk_dib = { CHAN_7909, 0, 0, 0, &dsk, &dsk_ini };

UNIT                dsk_unit[] = {
    {UDATA(&dsk_srv, UNIT_S_CHAN(4) | UNIT_DSK, 0), 0, 0x000, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(4) | UNIT_DSK, 0), 0, 0x102, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(4) | UNIT_DSK, 0), 0, 0x204, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(4) | UNIT_DSK, 0), 0, 0x306, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(4) | UNIT_DSK, 0), 0, 0x408, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(6) | UNIT_DSK, 0), 0, 0x500, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(6) | UNIT_DSK, 0), 0, 0x602, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(6) | UNIT_DSK, 0), 0, 0x704, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(6) | UNIT_DSK, 0), 0, 0x806, 0},
    {UDATA(&dsk_srv, UNIT_S_CHAN(6) | UNIT_DSK, 0), 0, 0x908, 0},
/* Second set for arm two of each unit */
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
/* Third set for arm one of second module */
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
/* Fourth set for arm two of second module */
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
    {UDATA(&dsk_srv, UNIT_DIS, 0), 0, 0xff, 0},
};

MTAB                dsk_mod[] = {
    {FORMAT_OK, 0, 0, "NOFORMAT", NULL, NULL, NULL},
    {FORMAT_OK, FORMAT_OK, "FORMAT", "FORMAT", NULL, NULL, NULL},
    {HA2_OK, 0, 0, "NOHA2", NULL, NULL, NULL},
    {HA2_OK, HA2_OK, "HA2", "HA2", NULL, NULL, NULL},
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "TYPE", "TYPE",
     &dsk_set_type, &dsk_get_type, NULL},
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "MODULE", "MODULE",
     &dsk_set_module, &dsk_get_module, NULL},
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "CHAN", "CHAN",
     &set_chan, &get_chan, NULL},
    {MTAB_XTD | MTAB_VUN | MTAB_VAL, 0, "SELECT", "SELECT",
     &chan9_set_select, &chan9_get_select, NULL},
    {0}
};

DEVICE              dsk_dev = {
    "DK", dsk_unit, NULL /* Registers */ , dsk_mod,
    NUM_DEVS_DSK, 8, 15, 1, 8, 36,
    NULL, NULL, &dsk_reset, &dsk_boot, NULL, NULL,
    &dsk_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};

uint32 dsk(UNIT * uptr, uint16 cmd, uint16 dev)
{
    /* If device is not active start it working */
    if (!sim_is_active(uptr))
	sim_activate(uptr, 10);
    return 1;
}

t_stat dsk_srv(UNIT * uptr)
{
    int                 chan;
    int                 sel;
    int                 schan;
    int                 dev = uptr->u3 & 0xff;
    int                 u = (uptr->u3 >> 8) & 0xf;
    UNIT               *base = &dsk_unit[u];

    chan = UNIT_G_CHAN(base->flags);
    sel = (base->flags & UNIT_SELECT) ? 1 : 0;
    schan = (chan * 2) + sel;
    /* Make sure channel is talking to us */
    if (sel != (sms[chan] & 1))
	goto seeking;
    /* Channel has disconnected, abort current operation. */
    if (chan_status[chan] & DEV_DISCO) {
	int                 i;

	/* Scan all units and stop them if they are on this channel */
	for (i = 0; i < NUM_DEVS_DSK; i++) {
	    int                 xchan = UNIT_G_CHAN(dsk_unit[i].flags);
	    int                 j;

	    if (xchan != chan)
		continue;
	    for (j = 3 * NUM_DEVS_DSK + i; j >= 0; j -= NUM_DEVS_DSK) {
		if (dsk_unit[j].u5 & DSKSTA_CMD) {
		    dsk_unit[j].u5 &=
			~(DSKSTA_CMD | DSKSTA_XFER | DSKSTA_SCAN);
		    sim_cancel(&dsk_unit[j]);
		}
	    }
	}
	cmd_buffer[chan] = 0;
	chan_status[chan] &= ~(DEV_DISCO | DEV_WEOR);
	sim_debug(DEBUG_CHAN, &dsk_dev, "unit=%d disconnecting\n\r", u);
	if (uptr->wait > 0)
	    sim_activate(uptr, 60);
	return SCPE_OK;
    }

    if (chan_sense[chan] & (CTL_SNS2)) {
	if ((chan_status[chan] & DEV_FULL) == 0) {
	    assembly[chan] = 0777777777777L & (sense_unit[schan] << 6);
	    sim_debug(DEBUG_SNS, &dsk_dev, "%d sense2 %012llo\n\r", u,
		      assembly[chan]);
	    chan_set(chan, DEV_FULL | DEV_REOR);
	} else {
	    chan_clear(chan, STA_SEL);
	}
	sense_unit[schan] = 0;
    }

    if (chan_sense[chan] & (CTL_SNS1)) {
	assembly[chan] = sense[schan];
	assembly[chan] |= sense_unit[schan] >> 30;
	sim_debug(DEBUG_SNS, &dsk_dev, "%d sense1 %012llo\n\r", u,
		  assembly[chan]);
	chan_set(chan, DEV_FULL);
	chan_sense[chan] &= ~CTL_SNS1;
	chan_sense[chan] |= CTL_SNS2;
	uptr->u5 |= DSKSTA_CMD;	/* So we catch disconnect */
    }

    if (chan_sense[chan] & CTL_CNTL)
	disk_cmd(uptr);

    /* Handle writing of data */
    if ((chan_sense[chan] & SNS_WRITE) && uptr->u5 & DSKSTA_CMD) {
	t_uint64            c;
	int                 eor = 0;
	int                 i;

	uptr->u5 |= DSKSTA_WRITE;	/* Flag as write */
	/* Check if we have a data yet */
	if ((chan_status[chan] & DEV_FULL) == 0) {
	    /* Flag as timming error */
	    disk_posterr(uptr, DATA_RESPONSE);
	    return SCPE_OK;
	}

	/* Move one word to buffer */
	c = assembly[chan];
	for (i = 30; i >= 0 && eor == 0; i -= 6)
	    eor = disk_write(uptr, (uint8)(c >> i & 077), chan, 0);

	/* Handle end of record */
	if (eor) {
	    chan_status[chan] |= DEV_REOR;
	    sense_unit[schan] |= ((t_uint64) 1L) << (unit_bit[dev] + 30);
	    if ((uptr->u5 & DSKSTA_CMSK) == DVSR &&
		(uptr->u5 & DSKSTA_XFER) == 0)
		sense[schan] |= PROG_NOREC;
	    uptr->u5 &= ~(DSKSTA_SCAN | DSKSTA_XFER);
	    cmd_buffer[chan] = 0;
	    chan_sense[chan] |= CTL_END;
	    chan9_set_attn(chan, sel);
	    chan_clear(chan, STA_SEL);
	}

	/* If last word, fill with zeros until we get eor */
	if (chan_status[chan] & DEV_WEOR && (uptr->u5 & DSKSTA_CHECK) == 0) {
	    while (!disk_write(uptr, 0x80, chan, 1)) ;
	    uptr->u5 &= ~(DSKSTA_SCAN | DSKSTA_XFER);
	    disk_wblock(uptr);	/* Flush block */
	}

	chan_clear(chan, DEV_FULL);
    }

    /* Handle reading of data */
    if ((chan_sense[chan] & SNS_READ) && uptr->u5 & DSKSTA_CMD) {
	t_uint64            c = 0;
	int                 eor = 0;
	int                 i;

	/* Check if we have a data yet */
	if ((chan_status[chan] & DEV_FULL)) {
	    /* Flag as timming error */
	    disk_posterr(uptr, DATA_RESPONSE);
	    return SCPE_OK;
	}

	/* Fill the buffer until end of record */
	for (i = 30; i >= 0 && eor == 0; i -= 6) {
	    uint8               data;

	    eor = disk_read(uptr, &data, chan);
	    c |= ((t_uint64) (data & 077)) << i;
	}
	assembly[chan] = c;
	chan_set(chan, DEV_FULL);
	if (eor) {
	    chan_status[chan] |= DEV_REOR;
	    sense_unit[schan] |= ((t_uint64) 1L) << (unit_bit[dev] + 30);
	    if ((uptr->u5 & DSKSTA_CMSK) == DVSR &&
		(uptr->u5 & DSKSTA_XFER) == 0)
		sense[schan] |= PROG_NOREC;
	    uptr->u5 &= ~(DSKSTA_SCAN | DSKSTA_XFER);
	    cmd_buffer[chan] = 0;
	    chan_sense[chan] |= CTL_END;
	    chan9_set_attn(chan, sel);
	    chan_clear(chan, STA_SEL);
	}
    }

  seeking:
    if (uptr->wait || uptr->u5 & DSKSTA_CMD)
	sim_activate(uptr, 30);

    /* Handle seeking */
    if (uptr->wait > 0) {
	uptr->wait--;
	if (uptr->wait == 0) {
	    sim_debug(DEBUG_EXP, &dsk_dev, "Seek done dev=%d\n", dev);
	    sense_unit[schan] |= ((t_uint64) 1L) << (unit_bit[dev] + 30);
	    chan9_set_attn(chan, sel);
	}
    }

    return SCPE_OK;
}

/* Post a error on a given unit. */
void
disk_posterr(UNIT * uptr, t_uint64 error)
{
    int                 dev = uptr->u3 & 0xff;
    int                 chan;
    int                 schan;
    int                 sel;
    int                 u = (uptr->u3 >> 8) & 0xf;

    uptr = &dsk_unit[u];
    chan = UNIT_G_CHAN(uptr->flags);
    sel = (uptr->flags & UNIT_SELECT) ? 1 : 0;
    schan = (chan * 2) + sel;
    chan_status[chan] |= DEV_REOR;
    sense[schan] &= STAT_SIXBIT;
    sense[schan] |= error;
    sense_unit[schan] |= ((t_uint64) 1L) << (unit_bit[dev] + 30);
    cmd_buffer[chan] = 0;
    chan_sense[chan] |= CTL_END;
    chan9_set_attn(chan, sel);
    if (error != 0)
	chan9_set_error(chan, SNS_UEND);
    chan_clear(chan, STA_SEL);
}

/* Post error for command that could not be completed */
void
disk_cmderr(UNIT * uptr, t_uint64 error)
{
    int                 chan;
    int                 schan;
    int                 sel;
    int                 mod;
    int                 u = (uptr->u3 >> 8) & 0xf;

    uptr = &dsk_unit[u];
    chan = UNIT_G_CHAN(uptr->flags);
    sel = (uptr->flags & UNIT_SELECT) ? 1 : 0;
    schan = (chan * 2) + sel;
    mod = (cmd_buffer[chan] >> 16) & 0x1f;
    chan_status[chan] |= DEV_REOR;
    sense[schan] &= STAT_SIXBIT;
    sense[schan] |= error;
    sense_unit[schan] |= ((t_uint64) 1L) << (unit_bit[mod] + 30);
    cmd_buffer[chan] = 0;
    chan_sense[chan] |= CTL_END;
    chan9_set_error(chan, SNS_UEND);
}

/* Process command */
void
disk_cmd(UNIT * uptr)
{
    t_uint64            c;
    UNIT               *base;
    int                 chan;
    int                 schan;
    int                 sel;
    int                 i;
    int                 mod;
    int                 u;

    /* Get information on unit */
    u = uptr - dsk_unit;
    base = &dsk_unit[(uptr->u3 >> 8) & 0xf];
    chan = UNIT_G_CHAN(base->flags);
    sel = (base->flags & UNIT_SELECT) ? 1 : 0;
    schan = (chan * 2) + sel;

    /* Check if we have a command yet */
    if ((chan_status[chan] & DEV_FULL) == 0) {
	disk_cmderr(uptr, DATA_RESPONSE);
	sim_activate(uptr, 10);
	return;
    }

    c = assembly[chan];
    chan_clear(chan, DEV_FULL);
    if (cmd_buffer[chan] == 0) {
	chan_sense[chan] &= ~(SNS_ATTN1 << sel);
	sense[chan] &= STAT_SIXBIT;
	/* Decode command to find actual module */
	for (i = 5; i >= 0; i--) {
	    int                  t;

	    t = (int)(c >> (i * 6)) & 0xf;
	    if (t != 012)
		cmd_buffer[chan] |= t << (i * 4);
	}
	mod = (cmd_buffer[chan] >> 8) & 0xff;
	/* 30, 24, 18, 12, 6, 0 */
	cmd_buffer[chan] <<= 8;
	sense[schan] &= STAT_SIXBIT;
	switch (cmd_buffer[chan] >> 24) {
	case DNOP:		/* Nop */
	case DREL:		/* Release */
	    /* Should not happen, but if read or write, cpy will be
	     * expected so turn on command flag to catch disconnect */
	    sim_debug(DEBUG_CMD, &dsk_dev, "unit=%d nop\n", u);
	    chan_set(chan, DEV_REOR);
	    uptr->u5 &= ~DSKSTA_CMSK;
	    cmd_buffer[chan] = 0;
	    break;

	case DEBM:		/* Eight Bit mode */
	    /* Should not happen, but if read or write, cpy will be
	     * expected so turn on command flag to catch disconnect */
	    sim_debug(DEBUG_CMD, &dsk_dev, "unit=$d eight bit mode\n", u);
	    sense[schan] &= ~STAT_SIXBIT;
	    chan_set(chan, DEV_REOR);
	    uptr->u5 &= ~DSKSTA_CMSK;
	    cmd_buffer[chan] = 0;
	    break;

	case DSBM:		/* Six bit mode */
	    /* Should not happen, but if read or write, cpy will be
	     * expected so turn on command flag to catch disconnect */
	    sim_debug(DEBUG_CMD, &dsk_dev, "unit=%d six bit mode\n", u);
	    sense[schan] |= STAT_SIXBIT;
	    chan_set(chan, DEV_REOR);
	    uptr->u5 &= ~DSKSTA_CMSK;
	    cmd_buffer[chan] = 0;
	    break;

	case DWRC:		/* Prepare to Write Check */
	case DVSR:		/* Prepare to Verify single record */
	case DWRF:		/* Prepare to Format */
	case DVTN:		/* Prepare to Verify track no addr */
	case DVCY:		/* Prepare to Verify Cyl */
	case DVTA:		/* Prepare to Verify track addr */
	case DVHA:		/* Pretare to Verify home addr */
	    /* Just ignore command if no read/write requested */
	    if ((chan_sense[chan] & (SNS_PWRITE | SNS_PREAD)) == 0) {
		sim_debug(DEBUG_EXP, &dsk_dev, "unit=%d Disk op no R/W\n",
			  u);
		chan_set(chan, DEV_REOR);
		cmd_buffer[chan] = 0;
		break;
	    }

	case DSAI:		/* Set Access Inoperative */
	case DSEK:		/* Seek */
	    /* Find actual owner of this command */
	    for (i = 0; i < NUM_DEVS_DSK; i++) {
		UNIT               *up;

		if ((dsk_unit[i].flags & (UNIT_SELECT | UNIT_CHAN)) !=
		    (uptr->flags & (UNIT_SELECT | UNIT_CHAN)))
		    continue;

		/* Adjust for unit */
		if ((0xff & dsk_unit[i].u3) == mod)
		    up = &dsk_unit[i];
		else if ((0xff & dsk_unit[i + NUM_DEVS_DSK].u3) == mod)
		    up = &dsk_unit[i + NUM_DEVS_DSK];
		else if ((0xff & dsk_unit[i + (NUM_DEVS_DSK * 2)].u3) == mod)
		    up = &dsk_unit[i + (NUM_DEVS_DSK * 2)];
		else if ((0xff & dsk_unit[i + (NUM_DEVS_DSK * 3)].u3) == mod)
		    up = &dsk_unit[i + (NUM_DEVS_DSK * 3)];
		else
		    continue;

		/* Check if unit is busy */
		if (up->u5 & DSKSTA_CMD || up->wait > 0) {
		    sim_debug(DEBUG_CMD, &dsk_dev, "unit=%d busy\n", u);
		    disk_cmderr(uptr, STAT_NOTRDY);
		} else {
		    uint8               t = cmd_buffer[chan] >> 24;

		    if (t == DWRC) {
			if (up->u5 & DSKSTA_CMSK && up->u5 & DSKSTA_WRITE) {
			    up->u5 |= DSKSTA_CHECK;
			} else {
			    disk_cmderr(up, PROG_INVSEQ);
			    return;
			}
		    } else {
			up->u5 &= ~(DSKSTA_CMSK | DSKSTA_CHECK
				    | DSKSTA_WRITE);
			up->u5 |= t;
		    }
		    sim_activate(up, 10);
		}
		return;
	    }

	    sim_debug(DEBUG_CMD, &dsk_dev, "unit=%d cmd=%06x\n", u,
		      cmd_buffer[chan]);
	    /* Flag as invalid command */
	    disk_cmderr(uptr, STAT_OFFLINE);
	    break;
	default:
	    sim_debug(DEBUG_CMD, &dsk_dev,
		      "unit=%d Unknown Command %06x\n\r", chan,
		      cmd_buffer[chan]);
	    /* Flag as invalid command */
	    disk_cmderr(uptr, PROG_INVCODE);
	    break;
	}
    } else {
	int                 t, ha;
	int                 trk;
	int                 cyl;

	/* Got second half of command so done */
	chan_set(chan, DEV_REOR);
	t = (int)(c >> (30)) & 0x3f;
	if (t != 012)
	    cmd_buffer[chan] |= t << 4;
	t = (int)(c >> (24)) & 0x3f;
	if (t != 012)
	    cmd_buffer[chan] |= t;

	/* Save command options */
	t = (int)(c >> 12) & 07777;
	uptr->u5 &= ~DSKSTA_ARGMSK;
	uptr->u5 |= t << DSKSTA_ARGSHFT;

	/* Check if there is a unit here */
	if ((base->flags & UNIT_ATT) == 0) {
	    disk_cmderr(uptr, STAT_OFFLINE);
	    return;
	}

	/* Find out track and cylinder of this operation */
	trk = bcd_to_track(cmd_buffer[chan]);

	/* Do command */
	switch (uptr->u5 & DSKSTA_CMSK) {
	case DSAI:		/* Set Access Inoperative */
	    detach_unit(uptr);
	case DNOP:		/* Nop */
	case DREL:		/* Release */
	case DEBM:		/* Eight Bit mode */
	case DSBM:		/* Six bit mode */
	    uptr->u5 &=
		~(DSKSTA_CMSK | DSKSTA_CMD | DSKSTA_CHECK | DSKSTA_WRITE);
	    cmd_buffer[chan] = 0;
	    break;

	case DSEK:		/* Seek */
	    cyl = trk / disk_type[base->u4].track;
	    if (cyl > disk_type[base->u4].cyl) {
		disk_cmderr(uptr, PROG_INVADDR);
		seek_addr[schan] = 0xffff;
		return;
	    }
	    t = cyl - arm_cyl[u];
	    uptr->u5 &=
		~(DSKSTA_CMSK | DSKSTA_CMD | DSKSTA_CHECK | DSKSTA_WRITE);
	    cmd_buffer[chan] = 0;
	    chan_sense[chan] &= ~SNS_UEND;
	    if (t < 0)
		t = -t;
	    sim_debug(DEBUG_CMD, &dsk_dev,
		      "unit=%d DSEK %d cylinders to %d trk=%d\n", u, t,
		      cyl, trk);
	    if (t == 0) {	/* At cylinder, give attention */
		uptr->wait = 1;	/* Electronic select time */
		break;
	    }
	    dtrack[u] = 077777;	/* Mark as invalid */
	    uptr->wait = 0;
	    /* From documentation, it looks like seeks were a fixed time
	     * based on movement between cylinder groups */
	    if (t > 50)
		uptr->wait = 180;
	    else if (t > 10)
		uptr->wait = 120;
	    else
		uptr->wait = 30;
	    uptr->wait *= 500 / 30;
	    arm_cyl[u] = cyl;
	    break;

	case DWRF:		/* Prepare to Format */
	    /* Verify ok to format */
	    if ((base->flags & FORMAT_OK) == 0) {
		disk_cmderr(uptr, PROG_FMTCHK);
		return;
	    }

	    cyl = trk / disk_type[base->u4].track;

	    /* Make sure positioned to correct track */
	    if (arm_cyl[u] != cyl) {
		disk_cmderr(uptr, PROG_INVSEQ);
		return;
	    }

	    fmt_cyl[u] = cyl;
	    sim_debug(DEBUG_CMD, &dsk_dev, "unit=%d: FMT %06x\n",
		      u, cmd_buffer[chan]);
	    uptr->u5 |= DSKSTA_SCAN | DSKSTA_CMD;
	    uptr->u6 = 0;
	    chan_set(chan, STA_SEL);
	    break;

	case DVHA:		/* Prepare to Verify home addr */
	    /* Verify HA2 ok to write */
	    if ((base->flags & HA2_OK) == 0 &&
		(chan_sense[chan] & SNS_PWRITE) != 0) {
		disk_cmderr(uptr, PROG_FMTCHK);
		return;
	    }

	case DVTA:		/* Prepare to Verify track addr */
	case DVTN:		/* Prepare to Verify track no addr */
	case DVCY:		/* Prepare to Verify Cyl */
	    /* Compute cylinder to use */
	    cyl = trk / disk_type[base->u4].track;

	    /* Make sure positioned to correct track */
	    if (arm_cyl[u] != cyl) {
		disk_cmderr(uptr, PROG_INVSEQ);
		return;
	    }

	    /* Make sure we have correct format for this 
	     * cylinder in buffer */
	    disk_rblock(uptr, trk);

	    sim_debug(DEBUG_CMD, &dsk_dev, "unit=%d Execute VTA %06x ",
		      u, cmd_buffer[chan]);

	    /* Verify that the home address matches */
	    if ((uptr->u5 & DSKSTA_CMSK) != DVHA) {
		ha = (077 & dbuffer[u][0]) << 6;
		ha |= 077 & dbuffer[u][1];
		/* Mask out bits we ignore */
		t &= 01717;
		ha &= 01717;
		/* Convert BCD 0 to Binary 0 */
		if ((ha & 07700) == 01200)
		    ha &= 077;
		if ((ha & 077) == 012)
		    ha &= 07700;
		if ((t & 07700) == 01200)
		    t &= 077;
		if ((t & 077) == 012)
		    t &= 07700;

		sim_debug(DEBUG_CMD, &dsk_dev, "HA %04o(c) %04o(d)\n", t,
			  ha);
		if (ha != t) {
		    disk_cmderr(uptr, PROG_NOREC);
		    return;
		}
	    } else {
		sim_debug(DEBUG_CMD, &dsk_dev, "HA ignored\n");
	    }

	    /* Start actual operations */
	    uptr->u5 |= DSKSTA_SCAN | DSKSTA_CMD;
	    uptr->u6 = 0;
	    chan_set(chan, STA_SEL);
	    break;

	case DVSR:		/* Prepare to Verify single record */
	    /* Make sure we have correct format for this 
	     * cylinder in buffer */
	    disk_rblock(uptr, trk);
	    /* Start actual operations */
	    sim_debug(DEBUG_CMD, &dsk_dev, "unit=%d DVSR %06x %06o\n",
		      u, cmd_buffer[chan], t);
	    uptr->u5 |= DSKSTA_SCAN | DSKSTA_CMD;
	    uptr->u6 = 0;
	    chan_set(chan, STA_SEL);
	    break;
	}
    }
    return;
}

int
disk_rblock(UNIT * uptr, int trk)
{
    int                 u = uptr - dsk_unit;
    struct disk_t      *dsk = &disk_type[uptr->u4];
    UNIT               *base = &dsk_unit[(uptr->u3 >> 8) & 0xf];
    FILE               *f;
    int                 offset = 0;
    int                 fbase = 0;

    if (u > (NUM_DEVS_DSK * 2)) {
	if (u > NUM_DEVS_DSK * 3) {
	    offset = 3 * dsk->cyl * dsk->track * dsk->bpt;
	    fbase = 3 * dsk->fmtsz;
	} else {
	    offset = 2 * dsk->cyl * dsk->track * dsk->bpt;
	    fbase = 2 * dsk->fmtsz;
	}
    } else {
	if (u > NUM_DEVS_DSK) {
	    offset = dsk->cyl * dsk->track * dsk->bpt;
	    fbase = dsk->fmtsz;
	}
    }
    f = base->fileref;
    offset += dsk->fmtsz * dsk->mods;

    if (uptr->u5 & DSKSTA_DIRTY) {
	disk_wblock(uptr);
    }
    if (arm_cyl[u] != fmt_cyl[u]) {
	sim_fseek(f, fbase + arm_cyl[u] * dsk->fbpt, SEEK_SET);
	sim_fread(fbuffer[u], 1, dsk->fbpt, f);
	fmt_cyl[u] = arm_cyl[u];
    }
    /* Read in actualy track data */
    if (dtrack[u] != trk) {
	sim_debug(DEBUG_DETAIL, &dsk_dev, "unit=%d Read track %d\n", u,
		  trk);
	sim_fseek(f, offset + trk * dsk->bpt, SEEK_SET);
	if (sim_fread(dbuffer[u], 1, dsk->bpt, f) != dsk->bpt)
	    memset(dbuffer[u], 0, dsk->bpt);
	dtrack[u] = trk;
    }
    return 1;
}

int
disk_wblock(UNIT * uptr)
{
    int                 u = uptr - dsk_unit;
    struct disk_t      *dsk = &disk_type[uptr->u4];
    UNIT               *base = &dsk_unit[(uptr->u3 >> 8) & 0xf];
    FILE               *f;
    int                 offset = 0;

    if (u > (NUM_DEVS_DSK * 2)) {
	if (u > NUM_DEVS_DSK * 3) {
	    offset = 3 * dsk->cyl * dsk->track * dsk->bpt;
	} else {
	    offset = 2 * dsk->cyl * dsk->track * dsk->bpt;
	}
    } else {
	if (u > NUM_DEVS_DSK) {
	    offset = dsk->cyl * dsk->track * dsk->bpt;
	}
    }
    f = base->fileref;
    offset += dsk->fmtsz * dsk->mods;

    if ((uptr->u5 & DSKSTA_DIRTY) == 0)
	return 1;

    /* Check if new format data */
    if ((uptr->u5 & DSKSTA_CMSK) == DWRF) {
	int                 s = disk_format(uptr, f, arm_cyl[u], base);

	if (s == 0)
	    disk_posterr(uptr, PROG_FMTCHK);
	return 1;
    }
    sim_debug(DEBUG_DETAIL, &dsk_dev, "unit=%d Write track %d\n",
	      u, dtrack[u]);
    /* Write in actualy track data */
    sim_fseek(f, offset + dtrack[u] * dsk->bpt, SEEK_SET);
    sim_fwrite(dbuffer[u], 1, dsk->bpt, f);
    uptr->u5 &= ~DSKSTA_DIRTY;
    return 1;
}

/* Convert a format pattern into a format track */
int
disk_format(UNIT * uptr, FILE * f, int cyl, UNIT * base)
{
    uint8               tbuffer[MAXTRACK];
    struct disk_t      *dsk = &disk_type[uptr->u4];
    int                 i, j;
    int                 out = 0;
    int                 u = uptr - dsk_unit;
    uint8               ch;
    int                 offset;

    f = base->fileref;

    offset = 0;
    if (u > (NUM_DEVS_DSK * 2)) {
	if (u > NUM_DEVS_DSK * 3) {
	    offset = 3 * dsk->fmtsz;
	} else {
	    offset = 2 * dsk->fmtsz;
	}
    } else {
	if (u > NUM_DEVS_DSK) {
	    offset = dsk->fmtsz;
	} else {
	    offset = 0;
	}
    }

    uptr->u5 &= ~(DSKSTA_DIRTY | DSKSTA_CHECK);
    sim_debug(DEBUG_DETAIL, &dsk_dev, "unit=%d format: ", u);
    /* Scan over new format */
    /* Convert format specification in place in dbuffer, then
     * pack it into fbuffer and write to file */

    /* Skip initial gap */
    for (i = 0; i < MAXTRACK && dbuffer[u][i] == 04; i++) ;
    if (i == MAXTRACK)
	return 0;		/* Failed if we hit end */
    /* HA1 Gap */
    for (j = i; i < MAXTRACK && dbuffer[u][i] == 03; i++) ;
    if ((i - j) > 12)
	return 0;		/* HA1 too big */
    if (dbuffer[u][i++] != 04)
	return 0;		/* Not gap */
    for (j = i; i < MAXTRACK && dbuffer[u][i] == 03; i++) ;
    if (i == MAXTRACK)
	return 0;		/* Failed if we hit end */
    if (dbuffer[u][i++] != 04)
	return 0;		/* Not gap */
    /* Size up HA2 gap */
    for (j = i;
	 i < MAXTRACK && (dbuffer[u][i] == 03 || dbuffer[u][i] == 01);
	 i++) ;
    j = i - j;
    if (j < 6)
	return 0;
    j -= dsk->overhd;		/* Remove overhead */
    sim_debug(DEBUG_DETAIL, &dsk_dev, "HA2(%d) ", j);
    for (; j > 0; j--)
	tbuffer[out++] = FMT_HA2;
    /* Now grab records */
    while (i < MAXTRACK) {
	ch = dbuffer[u][i++];
	if (ch == 0x80)
	    break;		/* End of record */
	if (ch != 04 && ch != 02)
	    return 0;		/* Not a gap */
	for (j = i; i < MAXTRACK && dbuffer[u][i] == ch; i++) ;
	ch = dbuffer[u][i];	/* Should be RA */
	/* Gap not long enough or eor */
	if (ch == 0x80 || (i - j) < 11)
	    break;
	/* Size up RA gap */
	if (ch != 01 && ch != 03)
	    return 0;		/* Not header */
	for (j = i; i < MAXTRACK && dbuffer[u][i] == ch; i++) ;
	j = i - j;
	if (j < 10)
	    return 0;
	j -= dsk->overhd;	/* Remove overhead */
	sim_debug(DEBUG_DETAIL, &dsk_dev, "RA(%d) ", j);
	for (; j > 0; j--)
	    tbuffer[out++] = FMT_HDR;
	ch = dbuffer[u][i++];
	if (ch != 04 && ch != 02)
	    return 0;		/* End of RA Field */
	ch = dbuffer[u][i];
	if (ch != 01 && ch != 03)
	    return 0;		/* Not gap */
	for (j = i; i < MAXTRACK && dbuffer[u][i] == ch; i++) ;
	if ((i - j) < 10)
	    return 0;		/* Gap not large enough */
	ch = dbuffer[u][i++];
	if (ch != 04 && ch != 02)
	    return 0;		/* End of RA Gap */
	ch = dbuffer[u][i];	/* Should be RA */
	if (ch != 01 && ch != 03)
	    return 0;		/* Not Record data */
	for (j = i; i < MAXTRACK && dbuffer[u][i] == ch; i++) ;
	j = i - j;
	if (j < 10)
	    return 0;
	j -= dsk->overhd;	/* Remove overhead */
	sim_debug(DEBUG_DETAIL, &dsk_dev, "DA(%d) ", j);
	for (; j > 0; j--)
	    tbuffer[out++] = FMT_DATA;
    }
    sim_debug(DEBUG_DETAIL, &dsk_dev, "total=%d\n", out);
    /* Put four END chars to end of buffer */
    for (j = 4; j > 0; j--)
	tbuffer[out++] = FMT_END;

    /* Make sure we did not pass size of track */
    if (dsk->fbpt > out)
	return 0;		/* Too big for track */

    /* Now grab every four characters and place them in next format location */
    for (j = i = 0; j < out; i++) {
	uint8               temp;

	temp = (tbuffer[j++] & 03);
	temp |= (tbuffer[j++] & 03) << 2;
	temp |= (tbuffer[j++] & 03) << 4;
	temp |= (tbuffer[j++] & 03) << 6;
	fbuffer[u][i] = temp;
    }

    /* Now write the buffer to the file */
    sim_fseek(f, offset + cyl * dsk->fbpt, SEEK_SET);
    sim_fwrite(fbuffer[u], 1, dsk->fbpt, f);
    return 1;
}

/* Handle writing of one character to disk */
int
disk_write(UNIT * uptr, uint8 data, int chan, int eor)
{
    int                 u = uptr - dsk_unit;
    int                 skip = 1;
    int                 flag = -1;

    while (skip) {
	flag = fbuffer[u][uptr->u6 / 4];
	flag >>= (uptr->u6 & 03) * 2;
	flag &= 03;
	switch (uptr->u5 & DSKSTA_CMSK) {
	case DWRF:		/* Format */
	    if ((uint32)uptr->u6 > disk_type[uptr->u4].bpt) {
		return 1;
	    }
	    if (uptr->u5 & DSKSTA_CHECK) {
		if (dbuffer[u][uptr->u6++] != data)
		    disk_posterr(uptr, DATA_CHECK);
	    } else {
		dbuffer[u][uptr->u6++] = data;
		uptr->u5 |= DSKSTA_DIRTY;
	    }
	    return 0;
	case DVTN:		/* Verify track no addr */
	    if (flag == FMT_END)
		return 1;
	    if (flag == FMT_DATA)
		skip = 0;
	    else
		uptr->u6++;
	    break;
	case DVTA:		/* Verify track addr */
	    if (flag == FMT_END)
		return 1;
	    if (flag != FMT_HA2)
		skip = 0;
	    else
		uptr->u6++;
	    break;
	case DVHA:		/* Verify home addr */
	    if (flag == FMT_END)
		return 1;
	    skip = 0;
	    break;
	case DVCY:		/*  Verify Cyl */
	    if (flag == FMT_END) {
		/* Move to next track */
		int                 trk = dtrack[u];
		int                 cyl = trk / disk_type[uptr->u4].track;

		if (eor)
		    return 1;

		if ((++trk) / disk_type[uptr->u4].track != cyl)
		    return 1;
		disk_rblock(uptr, trk);
		uptr->u6 = 0;
	    } else if (flag != FMT_DATA)
		uptr->u6++;
	    else
		skip = 0;
	    break;
	case DVSR:		/* Verify single record */
	    if (flag == FMT_DATA && uptr->u5 & DSKSTA_XFER) {
		skip = 0;
	    } else if (uptr->u5 & DSKSTA_XFER) {
		/* Not in a Data record, and we were transfering, all done */
		uptr->u5 &= ~DSKSTA_XFER;
		return 1;
	    } else if (flag == FMT_END) {
		/* If we hit the end, then no record found */
		disk_posterr(uptr, PROG_NOREC);
		return 1;
	    } else if (flag == FMT_HDR) {
		uint8               ch;
		uint16              first = 0;
		uint16              last = 0;
		int                 i;

		for (i = 0; i < 4 && flag == FMT_HDR; i++) {
		    ch = dbuffer[u][uptr->u6++];
		    first <<= 4;
		    if (ch != 012)
			first |= ch & 0xf;
		    flag = fbuffer[u][uptr->u6 / 4];
		    flag >>= (uptr->u6 & 03) * 2;
		    flag &= 03;
		}
		/* Short header, skip it, but should not have been made */
		if (flag != FMT_HDR)
		    break;
		/* Grab last two Characters */
		while (flag == FMT_HDR) {
		    ch = dbuffer[u][uptr->u6++];
		    last <<= 6;
		    last |= ch & 077;
		    flag = fbuffer[u][uptr->u6 / 4];
		    flag >>= (uptr->u6 & 03) * 2;
		    flag &= 03;
		}
		if (flag != FMT_DATA)
		    break;
		last &= 07777;
		/* Compare values. */
		if (first != (cmd_buffer[chan] & 0xffff))
		    break;
		if (last != ((uptr->u5 & DSKSTA_ARGMSK) >> DSKSTA_ARGSHFT))
		    break;
		/* Got a match, exit loop */
		uptr->u5 &= ~DSKSTA_SCAN;
		uptr->u5 |= DSKSTA_XFER;
		skip = 0;
	    } else {
		uptr->u6++;
	    }
	    break;
	}
    }
    data &= 077;		/* Format sets bit 7 to mark end of track */
    if (uptr->u5 & DSKSTA_CHECK) {
	if (dbuffer[u][uptr->u6++] != data)
	    disk_posterr(uptr, DATA_CHECK);
    } else {
	dbuffer[u][uptr->u6++] = data;
	uptr->u5 |= DSKSTA_DIRTY;
    }
    return 0;
}

/* Handle reading of one character to disk */
int
disk_read(UNIT * uptr, uint8 * data, int chan)
{
    int                 u = uptr - dsk_unit;
    int                 skip = 1;
    int                 flag;

    while (skip) {
	flag = fbuffer[u][uptr->u6 / 4];
	flag >>= (uptr->u6 & 03) * 2;
	flag &= 03;
	switch (uptr->u5 & DSKSTA_CMSK) {
	case DWRF:		/* Format */
	    disk_posterr(uptr, PROG_FMTCHK);
	    return 0;
	case DVTN:		/* Verify track no addr */
	    if (flag == FMT_END)
		return 1;
	    if (flag == FMT_DATA)
		skip = 0;
	    else
		uptr->u6++;
	    break;
	case DVTA:		/* Verify track addr */
	    if (flag == FMT_END)
		return 1;
	    if (flag != FMT_HA2)
		skip = 0;
	    else
		uptr->u6++;
	    break;
	case DVHA:		/* Pretare to Verify home addr */
	    if (flag == FMT_END)
		return 1;
	    skip = 0;
	    break;
	case DVCY:		/* Verify Cyl */
	    if (flag == FMT_END) {
		/* Move to next track */
		int                 trk = dtrack[u];
		UNIT               *base =
		    (u > NUM_DEVS_DSK) ? &uptr[-NUM_DEVS_DSK] : uptr;
		int                 cyl = trk / disk_type[base->u4].track;

		if ((++trk) / disk_type[base->u4].track != cyl)
		    return 1;
		disk_rblock(uptr, trk);
		uptr->u6 = 0;
	    } else if (flag != FMT_DATA)
		uptr->u6++;
	    else
		skip = 0;
	    break;
	case DVSR:		/* Verify single record */
	    if (flag == FMT_DATA && uptr->u5 & DSKSTA_XFER) {
		skip = 0;
	    } else if (uptr->u5 & DSKSTA_XFER) {
		/* Not in a Data record, and we were transfering, all done */
		uptr->u5 &= ~DSKSTA_XFER;
		return 1;
	    } else if (flag == FMT_END) {
		/* If we hit the end, then no record found */
		disk_posterr(uptr, PROG_NOREC);
		return 1;
	    } else if (flag == FMT_HDR) {
		uint8               ch;
		uint16              first = 0;
		uint16              last = 0;
		int                 i;

		for (i = 0; i < 4 && flag == FMT_HDR; i++) {
		    ch = dbuffer[u][uptr->u6++];
		    first <<= 4;
		    if (ch != 012)
			first |= ch & 0xf;
		    flag = fbuffer[u][uptr->u6 / 4];
		    flag >>= (uptr->u6 & 03) * 2;
		    flag &= 03;
		}
		/* Short header, skip it, but should not have been made */
		if (flag != FMT_HDR)
		    break;
		/* Grab last two Characters */
		while (flag == FMT_HDR) {
		    ch = dbuffer[u][uptr->u6++];
		    last <<= 6;
		    last |= ch & 077;
		    flag = fbuffer[u][uptr->u6 / 4];
		    flag >>= (uptr->u6 & 03) * 2;
		    flag &= 03;
		}
		if (flag != FMT_DATA)
		    break;
		last &= 07777;
		/* Compare values. */
		if (first != (cmd_buffer[chan] & 0xffff))
		    break;
		if (last != ((uptr->u5 & DSKSTA_ARGMSK) >> DSKSTA_ARGSHFT))
		    break;
		/* Got a match, exit loop */
		uptr->u5 &= ~DSKSTA_SCAN;
		uptr->u5 |= DSKSTA_XFER;
		skip = 0;
	    } else {
		uptr->u6++;
	    }
	    break;
	}
    }
    *data = dbuffer[u][uptr->u6++];
    return 0;
}

/* Convert BCD track address to binary address */
int
bcd_to_track(uint32 addr)
{
    int                 trk = 0;
    int                 i;

    for (i = 3; i >= 0; i--)
	trk = (trk * 10) + ((addr >> (i * 4)) & 0xf);
    return trk;
}

/* Boot disk. Build boot card loader in memory and transfer to it */
t_stat
dsk_boot(int unit_num, DEVICE * dptr)
{
    UNIT               *uptr = &dptr->units[unit_num];
    int                 chan = UNIT_G_CHAN(uptr->flags) - 1;
    int                 sel = (uptr->flags & UNIT_SELECT) ? 1 : 0;
    int                 dev = uptr->u3 & 0xff;
    int                 msk = (chan / 2) | ((chan & 1) << 11);

    if ((uptr->flags & UNIT_ATT) == 0)
	return SCPE_UNATT;	/* attached? */

    if (dev == 0)
	dev = 012;
    /* Build Boot program in memory */
    M[0] = 0000025000101L;	/*      IOCD RSCQ,,21 */
    M[1] = 0006000000001L;	/*      TCOA * */
    M[2] = 0002000000101L;	/*      TRA RSCQ */

    M[0101] = 0054000000115L;	/* RSCQ RSCC SMSQ  Mod */
    M[0101] |= ((t_uint64) (msk)) << 24;
    M[0102] = 0064400000000L;	/* SCDQ SCDC 0  Mod */
    M[0102] |= ((t_uint64) (msk)) << 24;
    M[0103] = 0044100000000L;	/*      LDI 0 */
    M[0104] = 0405400007100L;	/*      LFT 7100 */
    M[0105] = 0002000000110L;	/*      TRA *+3 */
    M[0106] = 0006000000102L;	/* TCOQ TCOC SCDQ  Mod */
    M[0106] |= ((t_uint64) (chan)) << 24;
    M[0107] = 0002000000003L;	/*      TRA 3    Enter IBSYS */
    M[0110] = 0076000000350L;	/* RICQ RICC **    Mod */
    M[0110] |= (chan + 1) << 9;
    M[0111] = 0500512001212L;	/*LDVCY DVCY  Mod */
    M[0111] |= ((t_uint64) (dev)) << 12;
    M[0112] = 0121222440000L;	/*      *    */
    M[0113] = 0501212001212L;	/*LDSEK DSEEK  Mod */
    M[0113] |= ((t_uint64) (dev)) << 12;
    M[0114] = 0121200000000L;	/*      *  */
    M[0115] = 0700000000016L;	/* SMSQ SMS   14 */
    M[0115] |= sel;
    M[0116] = 0200000000113L;	/*      CTL   LDSEK */
    M[0117] = 0500000200117L;	/*      TCM   *,,, */
    M[0120] = 0200000200111L;	/*      CTLR  LDVCY */
    M[0121] = 0400001000122L;	/*      CPYP  *+1,,1 */
    M[0122] = 0000000000122L;	/*      WTR * */
    M[0123] = 0100000000121L;	/*      TCH  *-2 */
    M[0124] = 0500000000000L;	/*      CPYD  0,,0 */
    M[0125] = 0340000000125L;	/*      TWT   * */
    IC = 0101;
    return SCPE_OK;
}

void
dsk_ini(UNIT * uptr, t_bool f)
{
    uptr->u5 = 0;
}

t_stat
dsk_reset(DEVICE * dptr)
{
    int                 i;
    int                 t;

    for (i = 0; i < NUM_CHAN; i++) {
	cmd_buffer[i] = 0;
	sense[i * 2] = sense[i * 2 + 1] = 0;
    }
    for (i = 0; i < NUM_DEVS_DSK; i++) {
	dtrack[i] = 077777;
	fmt_cyl[i] = 077777;
	arm_cyl[i] = 0;
	dtrack[i + NUM_DEVS_DSK] = 077777;
	fmt_cyl[i + NUM_DEVS_DSK] = 077777;
	arm_cyl[i + NUM_DEVS_DSK] = 0;
	dtrack[i + (NUM_DEVS_DSK * 2)] = 077777;
	fmt_cyl[i + (NUM_DEVS_DSK * 2)] = 077777;
	arm_cyl[i + (NUM_DEVS_DSK * 2)] = 0;
	dtrack[i + (NUM_DEVS_DSK * 3)] = 077777;
	fmt_cyl[i + (NUM_DEVS_DSK * 3)] = 077777;
	arm_cyl[i + (NUM_DEVS_DSK * 3)] = 0;
	t = dptr->units[i].u4;
	/* Fill in max capacity */
	dptr->units[i].capac = disk_type[t].mods * disk_type[t].bpt *
	    disk_type[t].arms * disk_type[t].track * disk_type[t].cyl;
	if (disk_type[t].arms > 1)
	    dptr->units[i + NUM_DEVS_DSK].u3 =
		0x10 | dptr->units[i].u3 | (i << 8);
	else
	    dptr->units[i + NUM_DEVS_DSK].u3 = 0xff;
    }
    return SCPE_OK;
}

/* Disk option setting commands */

t_stat
dsk_set_type(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    int                 i;

    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_ATT)
	return SCPE_ALATT;
    for (i = 0; disk_type[i].name != 0; i++) {
	if (strcmp(disk_type[i].name, cptr) == 0) {
	    uptr->u4 = i;
	    uptr[NUM_DEVS_DSK].u4 = i;
	    uptr->capac = disk_type[i].mods * disk_type[i].bpt *
		disk_type[i].arms * disk_type[i].track * disk_type[i].cyl;
	    return SCPE_OK;
	}
    }
    return SCPE_ARG;
}

t_stat
dsk_get_type(FILE * st, UNIT * uptr, int32 v, void *desc)
{
    if (uptr == NULL)
	return SCPE_IERR;
    fputs(disk_type[uptr->u4].name, st);
    return SCPE_OK;
}

t_stat
dsk_set_module(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    int                 u;

    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_ATT)
	return SCPE_ALATT;
    if (*cptr == '\0' || cptr[1] != '\0')
	return SCPE_ARG;
    if (*cptr < '0' || *cptr > '8')
	return SCPE_ARG;
    if (*cptr & 1)
	return SCPE_ARG;
    u = uptr->u3 & 0xf00;
    uptr->u3 = u | (*cptr - '0');
    if (disk_type[uptr->u4].arms > 1)
	uptr[NUM_DEVS_DSK].u3 = u | 0x10 | (*cptr - '0');
    else
	uptr[NUM_DEVS_DSK].u3 = 0xff;
    if (disk_type[uptr->u4].mods > 1) {
	uptr[NUM_DEVS_DSK * 2].u3 = (*cptr - '0') + 1;
	if (disk_type[uptr->u4].arms > 1)
	    uptr[NUM_DEVS_DSK * 3].u3 = u | 0x10 | ((*cptr - '0') + 1);
	else
	    uptr[NUM_DEVS_DSK * 3].u3 = 0xff;
    } else {
	uptr[NUM_DEVS_DSK * 2].u3 = 0xff;
	uptr[NUM_DEVS_DSK * 3].u3 = 0xff;
    }
    return SCPE_OK;
}

t_stat
dsk_get_module(FILE * st, UNIT * uptr, int32 v, void *desc)
{
    if (uptr == NULL)
	return SCPE_IERR;
    fprintf(st, "Module=%d", uptr->u3 & 0xff);
    return SCPE_OK;
}
