/* i7090_ht.c: ibm 7090 hypertape

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   Support for 7640 hypertape

   Magnetic tapes are represented as a series of variable records
   of the form:

        32b byte count
        byte 0
        byte 1
        :
        byte n-2
        byte n-1
        32b byte count

   If the byte count is odd, the record is padded with an extra byte
   of junk.  File marks are represented by a byte count of 0.

   Hypertape orders appear to be of the following formats. Since there
   is no documentation on them, I can going with what is shown in the
   IBSYS sources.

   BCD translated:
   CTLW/CTLR     06u01	    where u is unit numder
   CTLW/CTLR     07u01	    Backwords reading, where u is unit numder
   CTL           06uoo01    Where u is unit number, and oo is order code
				3x or 42
*/

#include "i7090_defs.h"
#include "sim_tape.h"

#define NUM_DEVS_HT	10
#define BUFFSIZE	(64 * 1024)

#define UNIT_HT(x)	UNIT_ATTABLE | UNIT_ROABLE | UNIT_S_CHAN(x)

/* in u3 is device address */
/* in u4 is current buffer position */
/* in u5 */
#define HT_CMDMSK   00000077	/* Command being run */
#define HT_NOTRDY   00000100	/* Devices is running a command */
#define HT_IDLE     00000200	/* Tape still in motion */
#define HT_MARK	    00000400	/* Hit tape mark */
#define HT_EOR      00001000	/* Hit end of record */
#define HT_ERR	    00002000	/* Device recieved error */
#define HT_BOT	    00004000	/* Unit at begining of tape */
#define HT_EOT	    00010000	/* Unit at end of tape */
#define HT_ATTN	    00020000	/* Unit requests attntion */
#define HT_MOVE	    00040000	/* Unit is moving to new record */
#define HT_WRITE    00100000	/* Were we writing */
#define HT_SNS      00200000	/* We were doing sense */
#define HT_CMD      00400000	/* We are fetching a command */
#define HT_PEND	    01000000	/* Hold channel while command runs */

/* Hypertape commands */
#define HNOP		0x00	/* Nop */
#define HEOS		0x01	/* End of sequence */
#define HRLF            0x02    /* Reserved Light Off */
#define HRLN            0x03    /* Reserved Light On */
#define HCLF            0x04    /* Check light off? Not documented by might
				   be valid command */
#define HCLN            0x05    /* Check light on */
#define HSEL		0x06	/* Select */
#define HSBR		0x07	/* Select for backwards reading */
#define HCCR		0x28	/* Change cartridge and rewind */
#define HRWD		0x30	/* Rewind */
#define HRUN		0x31	/* Rewind and unload */
#define HERG		0x32	/* Erase long gap */
#define HWTM		0x33	/* Write tape mark */
#define HBSR		0x34	/* Backspace */
#define HBSF		0x35	/* Backspace file */
#define HSKR		0x36	/* Space */
#define HSKF		0x37	/* Space file */
#define HCHC		0x38	/* Change Cartridge */
#define HUNL		0x39	/* Unload Cartridge */
#define HFPN		0x42	/* File Protect On */

uint32              ht(UNIT *, uint16, uint16);
t_stat              ht_srv(UNIT *);
t_stat              htc_srv(UNIT *);
void                ht_tape_cmd(DEVICE *, UNIT *);
t_stat              ht_error(UNIT *, int, t_stat);
t_stat              ht_boot(int32, DEVICE *);
t_stat              ht_reset(DEVICE *);
t_stat              ht_attach(UNIT *, char *);
t_stat              ht_detach(UNIT *);
extern uint16       IC;

/* One buffer per channel */
uint8               ht_unit[NUM_CHAN * 2];	/* Currently selected unit */
uint8               ht_buffer[NUM_CHAN][BUFFSIZE];
int                 ht_limit[NUM_CHAN];	/* End of buffer */
int                 ht_cmdbuffer[NUM_CHAN];	/* Buffer holding command ids */
int                 ht_cmdcount[NUM_CHAN];	/* Count of command digits recieved */
t_uint64            ht_sense[NUM_CHAN * 2];	/* Sense data for unit */
t_uint64            ht_sense2[NUM_CHAN * 2];	/* Sense word 2 data for unit */
DIB                 ht_dib = { CHAN_7909, 0, 0, 0, &ht, 0 };

UNIT                hta_unit[] = {
/* Controller 1 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 0 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 1 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 2 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 3 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 4 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 5 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 6 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 7 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 8 */
    {UDATA(&ht_srv, UNIT_HT(3), 0)},	/* 9 */
    {UDATA(&htc_srv, UNIT_DISABLE | UNIT_DIS, 0)},	/* Controller */
/* Controller 2 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 0 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 1 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 2 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 3 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 4 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 5 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 6 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 7 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 8 */
    {UDATA(&ht_srv, UNIT_HT(8), 0)},	/* 9 */
    {UDATA(&htc_srv, UNIT_DISABLE | UNIT_DIS, 0)},	/* Controller */
};

MTAB                ht_mod[] = {
    {MTUF_WLK, 0, "write enabled", "WRITEENABLED", NULL},
    {MTUF_WLK, MTUF_WLK, "write locked", "LOCKED", NULL},
    {MTAB_XTD | MTAB_VUN, 0, "FORMAT", "FORMAT",
     &sim_tape_set_fmt, &sim_tape_show_fmt, NULL},
    {MTAB_XTD | MTAB_VDV | MTAB_VAL, 0, "CHAN", "CHAN", &set_cchan, &get_chan,
     NULL},
    {MTAB_XTD | MTAB_VDV | MTAB_VAL, 0, "SELECT", "SELECT",
     &chan9_set_select, &chan9_get_select, NULL},
    {0}
};

DEVICE              hta_dev = {
    "HTA", hta_unit, NULL, ht_mod,
    NUM_DEVS_HT + 1, 8, 15, 1, 8, 36,
    NULL, NULL, &ht_reset, &ht_boot, &ht_attach, &ht_detach,
    &ht_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};
DEVICE              htb_dev = {
    "HTB", &hta_unit[NUM_DEVS_HT + 1], NULL, ht_mod,
    NUM_DEVS_HT + 1, 8, 15, 1, 8, 36,
    NULL, NULL, &ht_reset, &ht_boot, &ht_attach, &ht_detach,
    &ht_dib, DEV_DISABLE | DEV_DEBUG | DEV_DIS, 0, dev_debug
};

/* Hypertape sense word 1 codes */
#define SEL_MASK        0002700000000L	/* Selected unit mask */
#define STAT_NOTRDY	0200020000000L	/* Drive not ready */
#define PROG_NOTLOAD	0040004000000L	/* *Drive not loaded */
#define PROG_FILEPROT	0040002000000L	/* Drive write protected */
#define PROG_INVCODE	0040000200000L	/* Invalid code */
#define PROG_BUSY	0040000040000L	/* Drive Busy */
#define PROG_BOT	0040000020000L	/* Drive at BOT BSR/BSF requested */
#define PROG_EOT	0040000010000L	/* Drive at EOT forward motion requested. */
#define DATA_CHECK	0020000002000L	/* *Error corrected */
#define DATA_PARITY	0020000000400L	/* *Parity error */
#define DATA_CODECHK    0020000000200L	/* *Code check */
#define DATA_ENVCHK	0020000000100L	/* *Envelop error */
#define DATA_RESPONSE   0020000000020L	/* Response check */
#define DATA_EXECSKEW   0020000000004L	/* *Excessive skew check */
#define DATA_TRACKSKEW  0020000000002L	/* *Track skew check */
#define EXP_COND        0010000000000L

/* Word two, these all set Exception bit */
#define EXP_MSK		0260000000000L	/* Mask for exception conditions */
#define EXP_MARK	0200000000000L	/* Drive read a mark */
#define EXP_EWA		0040000000000L	/* *Drive near EOT */
#define EXP_NODATA	0020000000000L	/* *No data transfered */
#define READ_BSY	0002000000000L	/* *Controller reading */
#define WRITE_BSY	0000400000000L	/* *Controller writing */
#define BACK_MODE	0000200000000L	/* *Backwards mode */

uint32              ht(UNIT *, uint16, uint16);
t_stat              ht_srv(UNIT *);
t_stat              ht_reset(DEVICE *);
t_stat              ht_set_select(UNIT * uptr, int32 val, char *cptr,
				  void *desc);
t_stat              ht_get_select(FILE * st, UNIT * uptr, int32 v,
				  void *desc);
void                ht_tape_posterr(UNIT * uptr, t_uint64 error);
extern uint16       chan_sense[NUM_CHAN];

t_uint64            ht_unit_bit[] = {
    /*0   1   2   3   4   5   6   7   8   9 */
    /*A   4   2   1   A   4   2   1   A   4 */
    0000020000000L,		/* 1 */
    0000004000000L,		/* 2 */
    0000002000000L,		/* 3 */
    0000001000000L,		/* 4 */
    0000000200000L,		/* 5 */
    0000000040000L,		/* 6 */
    0000000020000L,		/* 7 */
    0000000010000L,		/* 8 */
    0000000002000L,		/* 9 */
    0000000000400L		/* 10 */
};

uint32 ht(UNIT * uptr, uint16 cmd, uint16 dev)
{
    DEVICE             *dptr = find_dev_from_unit(uptr);
    UNIT               *u = &dptr->units[NUM_DEVS_HT];

    /* Activate the device to start doing something */
    sim_activate(u, 10);
    return 1;
}

t_stat htc_srv(UNIT * uptr)
{
    DEVICE             *dptr = find_dev_from_unit(uptr);
    int                 chan = UNIT_G_CHAN(dptr->units[0].flags);
    int                 i;
    int                 sel;
    int                 schan;
    t_uint64            temp;

    sel = (dptr->units[0].flags & UNIT_SELECT) ? 1 : 0;
    schan = (chan * 2) + sel;
    if (chan_sense[chan] & (CTL_SNS1)) {
	int                 unit = ht_unit[schan];

	temp = ht_sense[schan];
	temp |= unit << 24;
	if (ht_sense2[schan] & EXP_MSK)
	    temp |= EXP_COND;
	assembly[chan] = temp;
	ht_sense[schan] = 0;
	sim_debug(DEBUG_SNS, dptr, "sense1 %012llo ", assembly[chan]);
	chan_set(chan, DEV_FULL);
	chan_sense[chan] &= ~CTL_SNS1;
	chan_sense[chan] |= CTL_SNS2;
	uptr->u5 |= HT_SNS;	/* So we catch disconnect */
	sim_activate(uptr, 10);
	return SCPE_OK;
    }

    if (chan_sense[chan] & (CTL_SNS2)) {
	if ((chan_status[chan] & DEV_FULL) == 0) {
	    UNIT               *up = dptr->units;

	    assembly[chan] = ht_sense2[schan];
	    ht_sense2[schan] = 0;
	    /* Grab all units who need attention */
	    for (i = 0; i < NUM_DEVS_HT; i++, up++) {
		if (up->u5 & HT_ATTN) {
		    assembly[chan] |= ht_unit_bit[i];
		    up->u5 &= ~HT_ATTN;
		}
	    }
	    /* Copy over tape mark */
	    up = &dptr->units[ht_unit[schan]];
	    if ((up->u5 & (HT_MARK | HT_NOTRDY)) == HT_MARK) {
//              assembly[chan] |= EXP_MARK;
		up->u5 &= ~HT_MARK;
	    }

	    sim_debug(DEBUG_SNS, dptr, "sense2 %012llo\n", assembly[chan]);
	    chan_set(chan, DEV_FULL | DEV_REOR);
	} else {
	    sim_debug(DEBUG_SNS, dptr, "sense2 ignored\n", assembly[chan]);
	    chan_clear(chan, STA_SEL);
	}
	sim_activate(uptr, 10);
	return SCPE_OK;
    }

    if (chan_sense[chan] & CTL_CNTL) {
	uptr->u5 |= HT_CMD;
	if (chan_status[chan] & DEV_WRITE) {
	    ht_tape_cmd(dptr, dptr->units);
	    sim_activate(uptr, 10);
	}
	return SCPE_OK;
    }

    /* Channel has disconnected, abort current operation. */
    if (chan_status[chan] & DEV_DISCO && uptr->u5 & HT_SNS) {
	uptr->u5 &= ~HT_SNS;
	chan_status[chan] &= ~(DEV_DISCO | DEV_WEOR);
	sim_debug(DEBUG_CHAN, dptr, "disconnecting\n");
    }
    return SCPE_OK;
}

t_stat ht_srv(UNIT * uptr)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    DEVICE             *dptr = find_dev_from_unit(uptr);
    int                 unit = (uptr - dptr->units);
    int                 i;
    int                 sel;
    int                 schan;
    t_uint64            temp;
    t_stat              r;

    sel = (uptr->flags & UNIT_SELECT) ? 1 : 0;
    schan = (chan * 2) + sel;

    /* Handle seeking */
    if (uptr->wait > 0) {
	uptr->wait--;
	if (uptr->wait == 0) {
	    if (uptr->u5 & HT_PEND)
		chan_set(chan, DEV_REOR);
	    else {
		uptr->u5 |= HT_ATTN;
		chan9_set_attn(chan, sel);
	    }
	    uptr->u5 &= ~(HT_PEND | HT_NOTRDY | HT_CMDMSK);
	    sim_debug(DEBUG_DETAIL, dptr, "%d Seek done\n", unit);
	} else
	    sim_activate(uptr, 500);
	return SCPE_OK;
    }

    /* Handle writing of data */
    if (chan_sense[chan] & SNS_WRITE &&
	(uptr->u5 & (HT_NOTRDY | HT_CMDMSK)) == (HT_NOTRDY | HSEL)) {
	int                 i;

	/* Check if we have a data yet */
	if ((chan_status[chan] & DEV_FULL) == 0) {
	    /* Nop flag as timming error */
	    ht_tape_posterr(uptr, DATA_RESPONSE);
	    return SCPE_OK;
	}
	uptr->u5 |= HT_WRITE;
	temp = assembly[chan];
	sim_debug(DEBUG_DATA, dptr, "Data=%012llo\n", temp);
	for (i = 30; i >= 0; i -= 6)
	    ht_buffer[chan][uptr->u6++] = (uint8)(temp >> i) & 077;
	if (uptr->u6 > BUFFSIZE || chan_status[chan] & DEV_WEOR) {
	    if (uptr->u6 != 0) {
		sim_debug(DEBUG_CMD, dptr,
			  " Write Block %d chars %d words\n", uptr->u6,
			  uptr->u6 / 6);
		r = sim_tape_wrrecf(uptr, &ht_buffer[chan][0], uptr->u6);
		uptr->u5 &= ~HT_WRITE;
		if (r != MTSE_OK) 
		    ht_error(uptr, schan, r);
	    }
	    uptr->u6 = 0;
	    chan9_set_attn(chan, sel);
	    chan_set(chan, DEV_REOR);
	    chan_clear(chan, STA_SEL);
	    uptr->u5 &= ~(HT_NOTRDY | HT_CMDMSK);
	}

	chan_clear(chan, DEV_FULL);
	sim_activate(uptr, 20);
	return SCPE_OK;
    }

    /* Handle reading of data */
    if (chan_sense[chan] & SNS_READ &&
	(uptr->u5 & (HT_NOTRDY | HT_CMDMSK)) == (HT_NOTRDY | HSEL)) {

	/* Check if we have a data yet */
	if ((chan_status[chan] & DEV_FULL)) {
	    /* Nop flag as timming error */
	    ht_tape_posterr(uptr, DATA_RESPONSE);
	    return SCPE_OK;
	}
	/* When we hit end of record transfer is done */
	if (uptr->u6 >= ht_limit[chan]) {
	    sim_debug(DEBUG_DATA, dptr, "EOD\n");
	    chan_sense[chan] |= CTL_END;
	    //chan9_set_attn(chan, sel);
	    chan_set(chan, DEV_REOR);
	    //chan_clear(chan, STA_SEL);
	    sim_activate(uptr, 20);
	    return SCPE_OK;
	}
	temp = 0;
	for (i = 30; i >= 0 && uptr->u6 <= ht_limit[chan]; i -= 6) {
	    temp |= (t_uint64) (ht_buffer[chan][uptr->u6++] & 077) << i;
	}
	assembly[chan] = temp;
	sim_debug(DEBUG_DATA, dptr, "Data=%012llo\n", temp);
	chan_set(chan, DEV_FULL);
	if (uptr->u6 > ht_limit[chan]) {
	    sim_debug(DEBUG_DATA, dptr, "Set EOR\n");
	    chan_set(chan, DEV_REOR);
	}
	sim_activate(uptr, 20);
	return SCPE_OK;
    }

    /* Channel has disconnected, abort current operation. */
    if (chan_status[chan] & DEV_DISCO && (uptr->u5 & HT_CMDMSK) == HSEL) {
	if (uptr->u5 & HT_WRITE) {
	    sim_debug(DEBUG_CMD, dptr,
		      "Write flush Block %d chars %d words\n", uptr->u6,
		      uptr->u6 / 6);
	    r = sim_tape_wrrecf(uptr, &ht_buffer[chan][0], uptr->u6);
	    uptr->u5 &= ~HT_WRITE;
	    if (r != MTSE_OK) {
		ht_error(uptr, schan, r);
		chan9_set_attn(chan, sel);
	    }
	    uptr->u6 = 0;
	}
	uptr->u5 &= ~(HT_NOTRDY | HT_CMDMSK);
	chan_clear(chan, DEV_DISCO | DEV_WEOR);
	sim_debug(DEBUG_CHAN, dptr, "disconnecting\n");
    }

    return SCPE_OK;
}

/* Post a error on a given unit. */
void
ht_tape_posterr(UNIT * uptr, t_uint64 error)
{
    int                 chan;
    int                 schan;
    int                 sel;

    chan = UNIT_G_CHAN(uptr->flags);
    sel = (uptr->flags & UNIT_SELECT) ? 1 : 0;
    schan = (chan * 2) + sel;
    uptr->u5 |= HT_ATTN;
    ht_sense[schan] = error;
    chan_sense[chan] |= CTL_END;
    chan_set(chan, DEV_REOR);
    chan9_set_attn(chan, sel);
    if (error != 0)
	chan9_set_error(chan, SNS_UEND);
    chan_clear(chan, STA_SEL);
}

/* Convert error codes to sense codes */
t_stat ht_error(UNIT * uptr, int schan, t_stat r)
{
    switch (r) {
    case MTSE_OK:		/* no error */
	break;

    case MTSE_TMK:		/* tape mark */
	uptr->u5 |= HT_MARK;
	ht_sense2[schan] |= EXP_MARK;
	break;

    case MTSE_WRP:		/* write protected */
	uptr->u5 |= HT_ATTN;
	ht_sense[schan] |= PROG_FILEPROT;
	break;

    case MTSE_UNATT:		/* unattached */
	uptr->u5 |= HT_ATTN;
	ht_sense[schan] = PROG_NOTLOAD;
	break;

    case MTSE_IOERR:		/* IO error */
    case MTSE_INVRL:		/* invalid rec lnt */
    case MTSE_FMT:		/* invalid format */
    case MTSE_RECE:		/* error in record */
	uptr->u5 |= HT_ERR;
	ht_sense[schan] |= DATA_CODECHK;
	break;
    case MTSE_BOT:		/* beginning of tape */
	uptr->u5 |= HT_BOT;
	ht_sense[schan] |= PROG_BOT;
	break;
    case MTSE_EOM:		/* end of medium */
	uptr->u5 |= HT_EOT;
	ht_sense[schan] |= PROG_EOT;
	break;
    default:			/* Anything else if error */
	ht_sense[schan] = PROG_INVCODE;
	break;
    }
    return SCPE_OK;
}

/* Process command */
void
ht_tape_cmd(DEVICE * dptr, UNIT * uptr)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    int                 sel = (uptr->flags & UNIT_SELECT) ? 1 : 0;
    int                 schan = (chan * 2) + sel;
    UNIT               *up;
    t_uint64            c;
    int                 i;
    int                 t;
    int                 cmd;
    int                 unit;
    int                 done;
    t_stat              r;
    t_mtrlnt            reclen;

    /* Get information on unit */

    /* Check if we have a command yet */
    if ((chan_status[chan] & DEV_FULL) == 0)
	return;

    c = assembly[chan];
    chan_clear(chan, DEV_FULL);
    done = 0;
    /* Decode command to find actual module */
    for (i = 30; i >= 0; i -= 6) {
	t = (uint8)(c >> i) & 0xf;
	if (t == 012)
	    t = 0;
	ht_cmdbuffer[chan] <<= 4;
	ht_cmdbuffer[chan] |= t;
	ht_cmdcount[chan]++;
	if ((ht_cmdbuffer[chan] & 0xff) == HEOS) {
	    uptr->u5 &= ~HT_CMD;
	    done = 1;
	    break;
	}
    }

    if (ht_cmdcount[chan] >= 8) {
	/* command error */
	ht_cmdcount[chan] = 0;
	ht_sense[schan] = PROG_INVCODE;
	ht_unit[schan] = 0;
	chan_sense[chan] |= SNS_UEND;
	chan_set(chan, DEV_REOR);
	uptr->u5 &= ~HT_CMD;
	return;
    }

    /* If done not set, did not find end of sequence, request more */
    if (done == 0)
	return;

    sim_debug(DEBUG_DETAIL, dptr, " cmd = %08x %d nybbles ",
	      ht_cmdbuffer[chan], ht_cmdcount[chan]);

    /* See if we have a whole command string yet */
    cmd = 0xff;
    unit = NUM_DEVS_HT + 1;
    for (i = ht_cmdcount[chan] - 2; i >= 2; i -= 2) {
	t = (ht_cmdbuffer[chan] >> (i * 4)) & 0xff;
	switch (t) {
	case HSEL:		/* Select */
	case HSBR:		/* Select for backwards reading */
	    i--;
	    unit = (ht_cmdbuffer[chan] >> (i * 4)) & 0xf;
	    ht_sense[schan] = 0;	/* Clear sense codes */
	    cmd = t;
	    break;
	case HEOS:		/* End of sequence */
	    break;
 	case HRLF:              /* Reserved Light Off */
 	case HRLN:              /* Reserved Light On */
	case HCLF:              /* Check light off */
	case HCLN:              /* Check light on */
	case HNOP:		/* Nop */
	case HCCR:		/* Change cartridge and rewind */
	case HRWD:		/* Rewind */
	case HRUN:		/* Rewind and unload */
	case HERG:		/* Erase long gap */
	case HWTM:		/* Write tape mark */
	case HBSR:		/* Backspace */
	case HBSF:		/* Backspace file */
	case HSKR:		/* Space */
	case HSKF:		/* Space file */
	case HCHC:		/* Change Cartridge */
	case HUNL:		/* Unload Cartridge */
	case HFPN:		/* File Protect On */
	    if (cmd != HSEL)
		cmd = 0xff;
	    else
		cmd = t;
	    break;
	default:
	    sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "Invalid command %x\n",
		  cmd);
	    ht_sense[schan] = PROG_INVCODE;
	    chan_set(chan, DEV_REOR);
	    chan9_set_error(chan, SNS_UEND);
	    return;
	}
    }

    /* Ok got a full command */
    ht_cmdcount[chan] = 0;

    /* Make sure we got a unit and command */
    if (unit <= NUM_DEVS_HT)
	ht_unit[schan] = unit;
    else {
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr,
		  "Invalid unit %d cmd=%x\n", unit, cmd);
	ht_sense[schan] = STAT_NOTRDY;
	chan_set(chan, DEV_REOR);
	chan9_set_error(chan, SNS_UEND);
	return;
    }

    if (cmd == 0xff) {
	/* command error */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "Invalid command %x\n",
		  cmd);
	ht_sense[schan] = PROG_INVCODE;
	chan_set(chan, DEV_REOR);
	chan9_set_error(chan, SNS_UEND);
	return;
    }

    /* Find real device this command is for */
    up = &uptr[unit];
    if ((up->flags & UNIT_ATT) == 0) {
	/* Not attached! */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "Not ready %d cmd=%x\n",
		  unit, cmd);
	ht_sense[schan] = STAT_NOTRDY;
	chan_set(chan, DEV_REOR);
	chan9_set_error(chan, SNS_UEND);
	return;
    }

    if (up->u5 & HT_NOTRDY || up->wait > 0) {
	/* Unit busy */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "Busy unit %d cmd=%x\n", unit,
		  cmd);
	ht_sense[schan] = PROG_BUSY;
	chan_set(chan, DEV_REOR);
	chan9_set_error(chan, SNS_UEND);
	return;
    }
    sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "Execute unit %d cmd=%x ",
	      unit, cmd);

    /* Ok, unit is ready and not in motion, set up to run command */
    up->u5 &= ~(HT_PEND | HT_MARK | HT_ERR | HT_CMDMSK);
    up->wait = 0;
    up->u5 |= t | HT_NOTRDY;
    r = MTSE_OK;
    switch (t) {
    case HSBR:			/* Select for backwards reading */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "HSBR\n");
	if (chan_sense[chan] & (SNS_PREAD)) {
	    r = sim_tape_rdrecr(up, &ht_buffer[chan][0], &reclen,
				BUFFSIZE);
	    if (r == MTSE_TMK)
	        sim_debug(DEBUG_CMD, dptr, "Read Mark\n");
	    else
	        sim_debug(DEBUG_CMD, dptr, "Read %d bytes %d words\n", reclen,
		      reclen / 6);
	    ht_limit[chan] = reclen;
	    /* Handle EOM special */
	    if (r == MTSE_EOM && (up->u5 & HT_EOT) == 0) {
		ht_sense2[schan] |= EXP_NODATA;
		up->u5 |= HT_EOT;
	    } else if (r == MTSE_OK) {
		ht_sense2[schan] |= BACK_MODE;
		/* Since we read buffer backwards ,
		   we can use same code to read it */
	        up->u5 &= ~HT_CMDMSK;
		up->u5 |= HSEL;
		chan_set(chan, STA_SEL);
	    }
	    up->u6 = 0;
	} else {
	    r = -1;
	}
	break;

    case HSEL:			/* Select */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "HSEL\n");
	ht_limit[chan] = -1;
	up->u6 = 0;
	if (chan_sense[chan] & SNS_PREAD) {
	    r = sim_tape_rdrecf(up, &ht_buffer[chan][0], &reclen,
				BUFFSIZE);
	    if (r == MTSE_TMK)
	        sim_debug(DEBUG_CMD, dptr, "Read Mark\n");
	    else
	        sim_debug(DEBUG_CMD, dptr, "Read %d bytes %d words\n", reclen,
		      reclen / 6);
	    ht_limit[chan] = reclen;
	    /* Handle EOM special */
	    if (r == MTSE_EOM && (up->u5 & HT_EOT) == 0) {
		up->u5 |= HT_EOT;
		ht_sense2[schan] |= EXP_NODATA;
	    } else if (r == MTSE_OK) {
		chan_set(chan, STA_SEL);
	    }
	} else if (chan_sense[chan] & SNS_PWRITE) {
	    if (sim_tape_wrp(up)) 
	        r = MTSE_WRP;
	    else
	        chan_set(chan, STA_SEL);
	} else {
	    r = -1;
	}
	break;

    case HRLF:                  /* Reserved Light Off */
    case HRLN:                  /* Reserved Light On */
    case HCLF:                  /* Check light off */
    case HCLN:                  /* Check light on */
    case HFPN:			/* File Protect On (Nop for now ) */
    case HEOS:			/* End of sequence */
    case HNOP:			/* Nop */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "NOP\n");
	up->u5 &= ~(HT_NOTRDY | HT_CMDMSK);
	chan_set(chan, DEV_REOR);
	chan_sense[chan] |= CTL_END;
	return;

    case HRWD:			/* Rewind */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "REW\n");
	if (up->u5 & HT_BOT) {
	    r = MTSE_OK;
	    up->wait = 1;
	} else {
	    r = sim_tape_rewind(up);
	    up->u5 |= HT_BOT;
	    up->u5 &= ~HT_EOT;
	    up->wait = 500;
	}
	break;

    case HERG:			/* Erase long gap */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "ERG\n");
	if (sim_tape_wrp(up)) {
	    r = MTSE_WRP;
	} else {
	    up->wait = 10;
	    up->u5 |= HT_PEND;
	    up->u5 &= ~HT_BOT;
	}
	break;

    case HWTM:			/* Write tape mark */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "WTM\n");
	if (sim_tape_wrp(up)) {
	    r = MTSE_WRP;
	} else {
	    r = sim_tape_wrtmk(up);
	    up->wait = 5;
	    up->u5 |= HT_PEND;
	    up->u5 &= ~(HT_BOT|HT_EOT);
	}
	break;

    case HBSR:			/* Backspace */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "BSR\n");
	if (sim_tape_bot(up)) {
	    r = MTSE_BOT;
	    break;
	}
	r = sim_tape_sprecr(up, &reclen);
	up->wait = reclen / 100;
	up->wait += 2;
	up->u5 |= HT_PEND;
	up->u5 &= ~(HT_BOT|HT_EOT);
	if (r == MTSE_TMK) {
	    r = MTSE_OK;
	    up->u5 |= HT_MARK;
	}
	if (sim_tape_bot(up))
	    up->u5 |= HT_BOT;
	else
	    up->u5 &= ~HT_BOT;
	break;

    case HBSF:			/* Backspace file */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "BSF\n");
	if (sim_tape_bot(up)) {
	    r = MTSE_BOT;
	    break;
	}
	while ((r = sim_tape_sprecr(up, &reclen)) == MTSE_OK) {
	    up->wait += reclen;
	}
	up->wait /= 100;
	up->wait += 2;
	up->u5 |= HT_PEND;
	up->u5 &= ~(HT_BOT|HT_EOT);
	if (r == MTSE_TMK) {
	    r = MTSE_OK;
	    up->u5 |= HT_MARK;
	}
	if (sim_tape_bot(up))
	    up->u5 |= HT_BOT;
	else
	    up->u5 &= ~HT_BOT;
	break;

    case HSKR:			/* Space */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "SKR\n");
	r = sim_tape_sprecf(up, &reclen);
	up->u5 |= HT_PEND;
	if (r == MTSE_TMK) {
	    r = MTSE_OK;
	    up->u5 |= HT_MARK;
	}
	up->wait = reclen / 100;
	up->wait += 2;
	up->u5 &= ~HT_BOT;
	break;

    case HSKF:			/* Space file */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "SKF\n");
	while ((r = sim_tape_sprecf(up, &reclen)) == MTSE_OK) {
	    up->wait += reclen;
	}
	up->wait /= 100;
	up->wait += 2;
	up->u5 |= HT_PEND;
	if (r == MTSE_TMK) {
	    r = MTSE_OK;
	    up->u5 |= HT_MARK;
	}
	up->u5 &= ~HT_BOT;
	break;

    case HCCR:			/* Change cartridge and rewind */
    case HRUN:			/* Rewind and unload */
    case HCHC:			/* Change Cartridge */
    case HUNL:			/* Unload Cartridge */
	sim_debug(DEBUG_DETAIL | DEBUG_CMD, dptr, "RUN\n");
	r = sim_tape_detach(up);
	up->wait = 100;
	break;
    }

    if (r != MTSE_OK) {
	chan_set(chan, DEV_REOR);
	ht_error(up, schan, r);
	chan9_set_error(chan, SNS_UEND);
	chan9_set_attn(chan, sel);
	up->u5 &= ~(HT_NOTRDY | HT_CMDMSK);
	up->wait = 0;
    } else if (up->u5 & HT_CMDMSK) {
	if (up->u5 & HT_PEND)
	    chan_clear(chan, DEV_WRITE);
	else
	    chan_set(chan, DEV_REOR);
	sim_activate(up, 500);
    } else {
	chan_set(chan, DEV_REOR);
	chan9_set_attn(chan, sel);
    }

    return;
}

/* Boot Hypertape. Build boot card loader in memory and transfer to it */
t_stat
ht_boot(int unit_num, DEVICE * dptr)
{
    UNIT               *uptr = &dptr->units[unit_num];
    int                 chan = UNIT_G_CHAN(uptr->flags) - 1;
    int                 sel = (uptr->flags & UNIT_SELECT) ? 1 : 0;
    int                 dev = uptr->u3;
    int                 msk = (chan / 2) | ((chan & 1) << 11);

    if ((uptr->flags & UNIT_ATT) == 0)
	return SCPE_UNATT;	/* attached? */

    if (dev == 0)
	dev = 012;
    /* Build Boot program in memory */
    M[0] = 0000025000101L;	/*      IOCD RSCQ,,21 */
    M[1] = 0006000000001L;	/*      TCOA * */
    M[2] = 0002000000101L;	/*      TRA RSCQ */

    M[0101] = 0054000000113L;	/* RSCQ RSCC SMSQ  Mod */
    M[0101] |= ((t_uint64) (msk)) << 24;
    M[0102] = 0064500000000L;	/* SCDQ SCDC 0  Mod */
    M[0102] |= ((t_uint64) (msk)) << 24;
    M[0103] = 0044100000000L;	/*      LDI 0 */
    M[0104] = 0405400001700L;	/*      LFT 1700 */
    M[0105] = 0002000000122L;	/*      TRA HYP7 */
    M[0106] = 0006000000102L;	/* TCOQ TCOC SCDQ  Mod */
    M[0106] |= ((t_uint64) (chan)) << 24;
    M[0107] = 0002000000003L;	/*      TRA 3    Enter IBSYS */
    M[0110] = 0120600120112L;
    M[0110] |= ((t_uint64) (dev)) << 18;
    M[0111] = 0120600030412L;	/*LDVCY DVCY  Mod */
    M[0111] |= ((t_uint64) (dev)) << 18;
    M[0112] = 0010000000000L;	/*      *    */
    M[0113] = 0700000000012L;	/* HYP6 SMS   10 */
    M[0113] |= sel;
    M[0114] = 0200000200110L;	/*      CTLR  *-4 */
    M[0115] = 0400001000116L;	/*      CPYP  *+1,,1 */
    M[0116] = 0000000000116L;	/*      WTR * */
    M[0117] = 0100000000115L;	/*      TCH  *-2 */
    M[0120] = 0700000400113L;	/*      SMS*  HYP6 */
    M[0121] = 0200000000111L;	/*      CTL  HYP6-2 */
    M[0122] = 0076000000350L;	/* HYP7 RICC **     */
    M[0122] |= ((t_uint64) (chan)) << 9;
    M[0123] = 0054000000120L;	/*      RSCC *-3  Mod */
    M[0123] |= ((t_uint64) (msk)) << 24;
    M[0124] = 0500000000000L;	/*      CPYD  0,,0 */
    M[0125] = 0340000000125L;	/*      TWT   * */
    IC = 0101;
    return SCPE_OK;
}

t_stat
ht_reset(DEVICE * dptr)
{
    int                 i;

    for (i = 0; i < NUM_DEVS_HT; i++) {
	ht_cmdbuffer[i] = ht_cmdcount[i] = 0;
	ht_sense[i] = 0;
	ht_sense2[i] = 0;
    }
    return SCPE_OK;
}

/* Disk option setting commands */
t_stat
ht_set_select(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    if (cptr == NULL)
	return SCPE_ARG;
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_ATT)
	return SCPE_ALATT;
    if (*cptr == '\0' || cptr[1] != '\0')
	return SCPE_ARG;
    if (*cptr == '0')
	uptr->flags &= ~UNIT_SELECT;
    else if (*cptr == '1')
	uptr->flags |= UNIT_SELECT;
    else
	return SCPE_ARG;
    return SCPE_OK;
}

t_stat
ht_get_select(FILE * st, UNIT * uptr, int32 v, void *desc)
{
    if (uptr == NULL)
	return SCPE_IERR;
    if (uptr->flags & UNIT_SELECT)
	fputs("Select=1", st);
    else
	fputs("Select=0", st);
    return SCPE_OK;
}

t_stat
ht_attach(UNIT * uptr, char *file)
{
    t_stat              r;

    if ((r = sim_tape_attach(uptr, file)) != SCPE_OK)
	return r;
    uptr->u5 = HT_BOT /*|HT_ATTN */ ;
    return SCPE_OK;
}

t_stat
ht_detach(UNIT * uptr)
{
    uptr->u5 = 0;
    if (uptr->flags & UNIT_DIS) return SCPE_OK;
    return sim_tape_detach(uptr);
}
