/* i7090_mt.c: IBM 7090 Magnetic tape controller

   Copyright (c) 2005, Richard Cornwell

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   Magnetic tapes are represented as a series of variable records
   of the form:

        32b byte count
        byte 0
        byte 1
        :
        byte n-2
        byte n-1
        32b byte count

   If the byte count is odd, the record is padded with an extra byte
   of junk.  File marks are represented by a byte count of 0.
*/

#include "i7090_defs.h"
#include "sim_tape.h"

#define NUM_UNITS_MT	4	/* A, B, C, D */
#define NUM_DEVS_MT	10
#define BUFFSIZE	(32 * 1024)

#define UNIT_MT(x)	UNIT_ATTABLE | UNIT_DISABLE | UNIT_ROABLE | \
			UNIT_S_CHAN(x)
#define MTUF_LDN	(1 << UNIT_V_LOCAL)
#define MTUF_ONLINE	(1 << (UNIT_V_LOCAL+1))

/* in u3 is device address */
/* in u4 is current buffer position */
/* in u5 */
#define MT_RDS		1
#define MT_RDSB		2
#define MT_WRS		3
#define MT_WRSB		4
#define MT_WEF		5
#define MT_BSR		6
#define MT_BSF		7
#define MT_REW		8
#define MT_SDN		9
#define MT_RUN		10
#define MT_SKIP		11	/* Do skip to end of record */
#define MT_WRITE	12	/* Actual transfer operation */
#define MT_CMDMSK   000017	/* Command being run */
#define MT_RDY 	    000020	/* Device is ready for command */
#define MT_IDLE     000040	/* Tape still in motion */
#define MT_MARK	    000100	/* Hit tape mark */
#define MT_EOT	    000200	/* At End Of Tape */

#define MTC_SEL	    0020	/* Controller executing read/write */
#define MTC_BSY     0040	/* Controller is busy - executing cmd */
#define MTC_UNIT    0017	/* device Channel is on */

uint32              mt(UNIT *, uint16, uint16);
t_stat              mt_srv(UNIT *);
t_stat              mt_boot(int32, DEVICE *);
void                mt_ini(UNIT *, t_bool);
t_stat              mt_reset(DEVICE *);
t_stat              mt_attach(UNIT *, char *);
t_stat              mt_detach(UNIT *);
t_stat              mt_rew(UNIT * uptr, int32 val, char *cptr, void *desc);
extern t_stat       chan_boot(int32, DEVICE *);
extern uint8        iocheck;
extern uint16       IC;

/* Channel level activity */
uint8               mt_chan[NUM_UNITS_MT];

/* One buffer per channel */
t_uint64            mt_buffer[NUM_UNITS_MT][BUFFSIZE];
DIB                 mt_dib =
    { CHAN_7607, NUM_DEVS_MT, 0201, 0740, &mt, &mt_ini };

UNIT                mta_unit[] = {
/* Controller 1 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 0 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 1 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 2 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 3 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 4 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 5 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 6 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 7 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 8 */
    {UDATA(&mt_srv, UNIT_MT(1), 0), 0},	/* 9 */
/* Controller 2 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 0 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 1 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 2 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 3 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 4 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 5 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 6 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 7 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 8 */
    {UDATA(&mt_srv, UNIT_MT(2), 0), 0},	/* 9 */
/* Controller 3 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 0 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 1 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 2 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 3 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 4 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 5 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 6 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 7 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 8 */
    {UDATA(&mt_srv, UNIT_MT(6), 0), 0},	/* 9 */
/* Controller 4 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 0 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 1 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 2 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 3 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 4 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 5 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 6 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 7 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 8 */
    {UDATA(&mt_srv, UNIT_MT(0), 0), 0},	/* 9 */
};

MTAB                mt_mod[] = {
    {MTUF_WLK, 0, "write enabled", "WRITEENABLED", NULL},
    {MTUF_WLK, MTUF_WLK, "write locked", "LOCKED", NULL},
    {MTUF_LDN, 0, "high density", "HIGH", NULL},
    {MTUF_LDN, MTUF_LDN, "low density", "LOW", NULL},
    {MTUF_ONLINE, 0, "offline", "OFFLINE", NULL},
    {MTUF_ONLINE, MTUF_ONLINE, "online", "ONLINE", NULL},
    {MTAB_XTD | MTAB_VUN, 0, "FORMAT", "FORMAT",
     &sim_tape_set_fmt, &sim_tape_show_fmt, NULL},
    {MTAB_XTD | MTAB_VUN, 0, NULL, "REWIND",
     &mt_rew, NULL, NULL},
    {MTAB_XTD | MTAB_VDV | MTAB_VAL, 0, "CHAN", "CHAN", &set_cchan, &get_chan,
     NULL},
    {0}
};

DEVICE              mta_dev = {
    "MTA", mta_unit, NULL, mt_mod,
    NUM_DEVS_MT, 8, 15, 1, 8, 36,
    NULL, NULL, &mt_reset, &mt_boot, &mt_attach, &mt_detach,
    &mt_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};
DEVICE              mtb_dev = {
    "MTB", &mta_unit[10], NULL, mt_mod,
    NUM_DEVS_MT, 8, 15, 1, 8, 36,
    NULL, NULL, &mt_reset, &mt_boot, &mt_attach, &mt_detach,
    &mt_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};
DEVICE              mtc_dev = {
    "MTC", &mta_unit[20], NULL, mt_mod,
    NUM_DEVS_MT, 8, 15, 1, 8, 36,
    NULL, NULL, &mt_reset, &mt_boot, &mt_attach, &mt_detach,
    &mt_dib, DEV_DISABLE | DEV_DEBUG, 0, dev_debug
};
DEVICE              mtd_dev = {
    "MTD", &mta_unit[30], NULL, mt_mod,
    NUM_DEVS_MT, 8, 15, 1, 8, 36,
    NULL, NULL, &mt_reset, &mt_boot, &mt_attach, &mt_detach,
    &mt_dib, DEV_DISABLE | DEV_DIS | DEV_DEBUG, 0, dev_debug
};

uint8               parity_table[64] = {
    /* 0    1    2    3    4    5    6    7 */
    0000, 0100, 0100, 0000, 0100, 0000, 0000, 0100,
    0100, 0000, 0000, 0100, 0000, 0100, 0100, 0000,
    0100, 0000, 0000, 0100, 0000, 0100, 0100, 0000,
    0000, 0100, 0100, 0000, 0100, 0000, 0000, 0100,
    0100, 0000, 0000, 0100, 0000, 0100, 0100, 0000,
    0000, 0100, 0100, 0000, 0100, 0000, 0000, 0100,
    0000, 0100, 0100, 0000, 0100, 0000, 0000, 0100,
    0100, 0000, 0000, 0100, 0000, 0100, 0100, 0000
};

/* Rewind tape drive */
t_stat
mt_rew(UNIT * uptr, int32 val, char *cptr, void *desc)
{
    /* If drive is offline or not attached return not ready */
    if ((uptr->flags & (UNIT_ATT | MTUF_ONLINE)) == 0)
	return SCPE_NOATT;
    /* Check if drive is ready to recieve a command */
    if ((uptr->u5 & MT_RDY) == 0)
	return STOP_IOCHECK;
    return sim_tape_rewind(uptr);
}

uint32 mt(UNIT * uptr, uint16 cmd, uint16 dev)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    DEVICE             *dptr = find_dev_from_unit(uptr);
    int                 time = 30;
    int                 unit = dev & 017;

    /* Make sure valid drive number */
    if (unit > NUM_DEVS_MT || unit == 0)
	return -1;
    unit--;			/* Adjust to origin zero */
    uptr += unit;
    /* If unit disabled return error */
    if (uptr->flags & UNIT_DIS) {
	fprintf(stderr, "Attempt to access disconnected unit %s%d\n\r",
		dptr->name, unit);
	return -1;
    }

    /* Check status of the drive */

    /* Can't do nothing if controller is busy */
    if (mt_chan[chan] & MTC_BSY)
	return 2;
    /* If drive is offline or not attached return not ready */
    if ((uptr->flags & (UNIT_ATT | MTUF_ONLINE)) !=
	(UNIT_ATT | MTUF_ONLINE)) {
	fprintf(stderr, "Attempt to access offline unit %s%d\n\r",
		dptr->name, unit);
	return 0;
    }
    /* Check if drive is ready to recieve a command */
    if ((uptr->u5 & MT_RDY) == 0) {
	/* Return indication if not ready and doing TRS */
	if (cmd == OP_TRS) {
	    return 0;
	} else
	    return 2;
    }
    uptr->u5 &= ~(MT_CMDMSK | MT_RDY);
    switch (cmd) {
    case OP_RDS:
	if (mt_chan[chan] & MTC_SEL) {
	    uptr->u5 |= MT_RDY;
	    return 2;
	}

	if (dev & 020)
	    uptr->u5 |= MT_RDSB;
	else
	    uptr->u5 |= MT_RDS;
	time = 350;
	if ((uptr->u5 & MT_IDLE) == 0)
	    time = 1350;
	if (sim_tape_bot(uptr))
	    time = 10000;
	chan_set_sel(chan, 0);
	chan_clear_status(chan);
	mt_chan[chan] &= MTC_BSY;
	mt_chan[chan] |= MTC_SEL | unit;
	uptr->u5 &= ~(MT_MARK);
	uptr->u6 = 0;
	sim_debug(DEBUG_CMD, dptr, "RDS unit=%d %o IC=%o\n", unit, dev,
		  IC);
	break;
    case OP_WRS:
	if (mt_chan[chan] & MTC_SEL) {
	    uptr->u5 |= MT_RDY;
	    return 2;
	}
	if (sim_tape_wrp(uptr)) {
	    sim_debug(DEBUG_EXP, dptr,
		      "WRS %d IC=%o attempted on locked tape\n", unit, IC);
	    uptr->u5 |= MT_RDY;
	    return -1;
	}
	if (dev & 020)
	    uptr->u5 |= MT_WRSB;
	else
	    uptr->u5 |= MT_WRS;
	uptr->u6 = 0;
	chan_set_sel(chan, 1);
	chan_clear_status(chan);
	mt_chan[chan] &= MTC_BSY;
	mt_chan[chan] |= MTC_SEL | unit;
	uptr->u5 &= ~(MT_MARK | MT_EOT);
	time = 500;
	if ((uptr->u5 & MT_IDLE) == 0)
	    time = 1350;
	if (sim_tape_bot(uptr))
	    time = 15000;
	sim_debug(DEBUG_CMD, dptr, "WRS unit=%d %o IC=%o\n", unit, dev,
		  IC);
	break;

    case OP_WEF:
	if (sim_tape_wrp(uptr)) {
	    sim_debug(DEBUG_EXP, dptr,
		      "WRS %d IC=%o attempted on locked tape\n", unit, IC);
	    uptr->u5 |= MT_RDY;
	    return 0;
	}
	if ((uptr->u5 & MT_IDLE) == 0)
	    time = 2000;
	uptr->u5 |= MT_WEF;
	uptr->u5 &= ~(MT_EOT);
	mt_chan[chan] |= MTC_BSY;
	sim_debug(DEBUG_CMD, dptr, "WEF unit=%d IC=%o\n", unit, IC);
	break;

    case OP_BSR:
	/* Check if at load point, quick return if so */
	if (sim_tape_bot(uptr)) {
	    sim_debug(DEBUG_CMD, dptr, "BSR unit=%d at BOT IC=%o\n", unit,
		      IC);
	    uptr->u5 |= MT_RDY;
	    chan_set(chan, CHS_BOT);
	    return 1;
	}
	uptr->u5 |= MT_BSR;
	uptr->u5 &= ~(MT_EOT);
	mt_chan[chan] |= MTC_BSY;
	sim_debug(DEBUG_CMD, dptr, "BSR unit=%d IC=%o\n", unit, IC);
	break;
    case OP_BSF:
	/* Check if at load point, quick return if so */
	if (sim_tape_bot(uptr)) {
	    sim_debug(DEBUG_CMD, dptr, "BSF unit=%d at BOT IC=%o\n", unit,
		      IC);
	    uptr->u5 |= MT_RDY;
	    chan_set(chan, CHS_BOT);
	    return 1;
	}
	/* Back space over EOF if at mark */
//      if (uptr->u5 & MT_MARK)
//           sim_tape_sprecr( uptr, &reclen);
	uptr->u5 &= ~(MT_EOT);
	uptr->u5 |= MT_BSF;
	mt_chan[chan] |= MTC_BSY;
	sim_debug(DEBUG_CMD, dptr, "BSF unit=%d IC=%o\n", unit, IC);
	break;
    case OP_REW:
	/* Check if at load point, quick return if so */
	if (sim_tape_bot(uptr)) {
	    sim_debug(DEBUG_CMD, dptr, "REW unit=%d at BOT IC=%o\n", unit,
		      IC);
	    uptr->u5 |= MT_RDY;
	    return 1;
	}
	uptr->u5 |= MT_REW;
	uptr->u5 &= ~(MT_EOT);
	mt_chan[chan] |= MTC_BSY;
	sim_debug(DEBUG_CMD, dptr, "REW unit=%d IC=%o\n", unit, IC);
	break;
    case OP_RUN:
	chan_clear_status(chan);
	uptr->u5 |= MT_RUN;
	mt_chan[chan] |= MTC_BSY;
	uptr->u5 &= ~(MT_EOT);
	sim_debug(DEBUG_CMD, dptr, "RUN unit=%d IC=%o\n", unit, IC);
	break;
    case OP_SDN:
	uptr->u5 |= MT_RDY;	/* Command is quick */
	if (dev & 020)
	    uptr->flags |= MTUF_LDN;
	else
	    uptr->flags &= ~MTUF_LDN;
	sim_debug(DEBUG_CMD, dptr, "SDN unit=%d %o IC=%o\n", unit, dev,
		  IC);
	return 1;
    case OP_DRS:
	uptr->flags &= ~MTUF_ONLINE;
	uptr->u5 |= MT_RDY;	/* Command is quick */
	sim_debug(DEBUG_CMD, dptr, "DRS unit=%d IC=%o\n", unit, IC);
	return 1;
    case OP_TRS:
	uptr->u5 |= MT_RDY;	/* Get here we are ready */
	sim_debug(DEBUG_CMD, dptr, "TRS unit=%d IC=%o\n", unit, IC);
	return 1;
    }
    sim_cancel(uptr);
    sim_activate(uptr, time);
    return 1;
}

t_stat
mt_read_buff(UNIT * uptr, int cmd, DEVICE * dptr)
{
    uint8               buffer[BUFFSIZE * 6];
    int                 chan = UNIT_G_CHAN(uptr->flags);
    t_mtrlnt            reclen;
    int                 words;
    t_uint64            wd = 0;
    t_stat              r;
    uint32              i;
	int					j;
    uint8               ch;
    int                 mode = 0;
    int                 mark = 0;
    int                 parity = 0;

    uptr->u6 = 0;		/* Set to no data */
    uptr->u5 &= ~MT_MARK;
    if (cmd == MT_RDS)
	mode = 1;
    if ((r = sim_tape_rdrecf(uptr, buffer, &reclen, BUFFSIZE * 6)) !=
	MTSE_OK) return r;
    uptr->u6 = words = reclen / 6;
    sim_debug(DEBUG_DETAIL, dptr, "%s Block %d chars %d words\n",
	      (mode) ? "BCD" : "Binary", reclen, words);
    if ((reclen - (words * 6)) != 0) {	/* Extra words */
	uptr->u6++;
	words++;
    }
    /* Copy data to buffer */
    for (j = 6, i = 0; i < reclen; i++) {
	ch = buffer[i] & 077;
	/* Do BCD translation */
	if (mode) {
	    if (parity_table[ch] != (buffer[i] & 0100)) {
		parity = 1;
	    }
	    ch ^= (ch & 020) << 1;
	    if (ch == 012)
		ch = 0;
	    if (ch == 017) {
		mark = 1;
		chan_set_error(chan);	/* Force CRC error. */
	    }
	} else {
	    if (parity_table[ch] == (buffer[i] & 0100)) {
		parity = 1;
	    }
	}
	wd |= ((t_uint64) ch) << (6 * --j);
	if (mark)
	    wd &= 077777777;
	if (j == 0) {
	    mt_buffer[chan][words--] = wd;
	    wd = 0;
	    j = 6;
	}
    }
    if (j != 6)
	mt_buffer[chan][words--] = wd;

    /* If short read, shift down in buffer */
    if (words != 0) {
	for (i = 1; words < uptr->u6;)
	    mt_buffer[chan][i++] = mt_buffer[chan][words++];
	uptr->u6 = i;
    }
    if (parity) {
	chan_set_error(chan);	/* Force redundency error */
    }

    return MTSE_OK;
}

t_stat
mt_write_buff(UNIT * uptr, int cmd, DEVICE * dptr)
{
    uint8               buffer[BUFFSIZE * 6];
    int                 chan = UNIT_G_CHAN(uptr->flags);
    t_mtrlnt            reclen;
    uint16              words;
    t_uint64            wd;
    int                 j;
    uint8               ch;
    int                 mode = 0;

    /* Don't write zero lenght records. */
    if (uptr->u6 == 0)
	return SCPE_OK;
    /* Copy data to buffer */
    reclen = 0;
    if (cmd == MT_WRS)
	mode = 1;
    for (words = 0; words < uptr->u6; words++) {
	wd = mt_buffer[chan][words];
	for (j = 5; j >= 0; j--) {
	    /* Do BCD translation */
	    ch = 077 & (uint8)(wd >> (6 * j));
	    if (mode) {
		ch ^= (ch & 020) << 1;
		if (ch == 0)
		    ch = 012;
		ch |= parity_table[ch];
	    } else {
		ch |= 0100 ^ parity_table[ch];
	    }
	    buffer[reclen++] = ch;
	}
    }
    uptr->u6 = 0;
    sim_debug(DEBUG_DETAIL, dptr, "%s Block %d chars %d words\n",
	      (mode) ? "BCD" : "Binary", reclen, words);
    return sim_tape_wrrecf(uptr, buffer, reclen);
}

t_stat mt_error(UNIT * uptr, int chan, t_stat r, DEVICE * dptr)
{
    switch (r) {
    case MTSE_OK:		/* no error */
	break;

    case MTSE_TMK:		/* tape mark */
	sim_debug(DEBUG_EXP, dptr, "MARK ");
	uptr->u5 |= MT_MARK;
	chan_set_eof(chan);
	break;

    case MTSE_WRP:		/* write protected */
    case MTSE_UNATT:		/* unattached */
	sim_debug(DEBUG_EXP, dptr, "ATTENTION %d ", r);
	iocheck = 1;
	chan_set_attn(chan);
	break;

    case MTSE_IOERR:		/* IO error */
    case MTSE_FMT:		/* invalid format */
    case MTSE_RECE:		/* error in record */
	chan_set_error(chan);	/* Force redundency error */
	chan_set_attn(chan);	/* Set error */
	sim_debug(DEBUG_EXP, dptr, "ERROR %d ", r);
	break;
    case MTSE_BOT:		/* beginning of tape */
	chan_set(chan, CHS_BOT);	/* Set flag */
	sim_debug(DEBUG_EXP, dptr, "BOT ", r);
	break;
    case MTSE_INVRL:		/* invalid rec lnt */
    case MTSE_EOM:		/* end of medium */
	uptr->u5 |= MT_EOT;
	sim_debug(DEBUG_EXP, dptr, "EOT ", r);
	break;
    }
    return SCPE_OK;
}

t_stat mt_srv(UNIT * uptr)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);
    DEVICE             *dptr = find_dev_from_unit(uptr);
    int                 unit = (uptr - dptr->units) & MTC_UNIT;
    int                 cmd = uptr->u5 & MT_CMDMSK;
    t_mtrlnt            reclen;
    t_stat              r;

    /* Channel has disconnected, abort current read. */
    if (chan_status[chan] & DEV_DISCO
	&& (mt_chan[chan] & 037) == (MTC_SEL | unit)) {
	uptr->u5 &= ~MT_CMDMSK;
	if (cmd == MT_WRS || cmd == MT_WRSB) {
	    sim_debug(DEBUG_DETAIL, dptr, "Write unit=%d ", unit);
	    mt_write_buff(uptr, cmd, dptr);
	    sim_activate(uptr, 5000);
	    mt_chan[chan] &= MTC_BSY;
	    uptr->u5 |= MT_RDY;
	} else if (cmd = MT_RDS || cmd == MT_RDSB) {
	    /* Keep moving until end of block */
	    if (uptr->u6 > 0) {
		uptr->u5 |= MT_SKIP;
		sim_activate(uptr, uptr->u6 * 40);
	    } else {
		sim_activate(uptr, 5000);
		uptr->u5 |= MT_RDY;
		mt_chan[chan] &= MTC_BSY;
	    }
	    uptr->u6 = -1;
	}
	sim_debug(DEBUG_CHAN, dptr, "Disconnect unit=%d", unit);
	uptr->u5 |= MT_IDLE;
	chan_clear(chan, DEV_DISCO | DEV_WEOR | STA_SEL);
	return SCPE_OK;
    }

    switch (uptr->u5 & MT_CMDMSK) {
    case 0:			/* No command, stop tape */
	uptr->u5 &= ~MT_IDLE;
	uptr->u5 |= MT_RDY;	/* Ready since command is done */
	return SCPE_OK;

    case MT_SKIP:		/* Record skip done, enable tape drive */
	uptr->u5 &= ~MT_CMDMSK;
	uptr->u5 |= MT_RDY | MT_IDLE;
	mt_chan[chan] &= MTC_BSY;	/* Clear all but busy */
	sim_activate(uptr, 5000);
	return SCPE_OK;

    case MT_RDS:
    case MT_RDSB:
	if (uptr->u6 == 0) {
	    sim_debug(DEBUG_DETAIL, dptr, "Read unit=%d ", unit);
	    if ((r = mt_read_buff(uptr, cmd, dptr)) != MTSE_OK) {
		uptr->u5 &= ~MT_CMDMSK;
		sim_activate(uptr, 100);
		chan_set_attn(chan);
		chan_set(chan, DEV_REOR);
		return mt_error(uptr, chan, r, dptr);
	    }
	}
	switch (chan_write(chan, &mt_buffer[chan][uptr->u6],
			   (uptr->u6 == 1) ? DEV_REOR : 0)) {
	case END_RECORD:
	    sim_debug(DEBUG_DATA, dptr, "Read unit=%d EOR\n", unit);
	    if (uptr->u6 != 0) {
		uptr->u5 &= ~MT_CMDMSK;
		uptr->u5 |= MT_SKIP;
		sim_activate(uptr, uptr->u6 * 40);
		chan_set(chan, DEV_REOR);
		uptr->u6 = 0;	/* Force read next record */
		break;
	    }
	case DATA_OK:
	    uptr->u6--;
	    sim_debug(DEBUG_DATA, dptr, "Read data unit=%d %d %012o\n\r",
		      unit, uptr->u6, mt_buffer[chan][uptr->u6 + 1]);
	    if (uptr->u6 == 0)	/* In IRG */
		sim_activate(uptr, 100);
	    else
		sim_activate(uptr, 45);
	    break;

	case TIME_ERROR:
	    uptr->u5 &= ~MT_CMDMSK;
	    uptr->u5 |= MT_SKIP;
	    sim_activate(uptr, uptr->u6 * 40);
	    uptr->u6 = 0;	/* Force read next record */
	    break;
	}

	return SCPE_OK;

    case MT_WRS:
    case MT_WRSB:
	switch (chan_read(chan, &mt_buffer[chan][uptr->u6],
			  (uptr->u6 > BUFFSIZE) ? DEV_WEOR : 0)) {
	case TIME_ERROR:
	    chan_set_attn(chan);
	case END_RECORD:
	    sim_debug(DEBUG_DETAIL, dptr, "Write unit=%d", unit);
	    mt_write_buff(uptr, cmd, dptr);
	    break;
	case DATA_OK:
	    sim_debug(DEBUG_DATA, dptr, "Write data unit=%d %d %012o\n\r",
		      unit, uptr->u6, mt_buffer[chan][uptr->u6 + 1]);
	    uptr->u5 &= ~MT_MARK;
	    uptr->u6++;
	    break;
	}
	sim_activate(uptr, 45);
	return SCPE_OK;

    case MT_WEF:
	sim_debug(DEBUG_DETAIL, dptr, "Write Mark unit=%d\n", unit);
	uptr->u5 &= ~(MT_CMDMSK | MT_MARK);
	uptr->u5 |= (MT_RDY | MT_IDLE);
	r = sim_tape_wrtmk(uptr);
	mt_chan[chan] &= ~MTC_BSY;
	sim_activate(uptr, 5000);
	break;

    case MT_BSR:
	sim_debug(DEBUG_DETAIL, dptr, "Backspace rec unit=%d\n", unit);
	/* Clear tape mark, command, idle since we will need to change dir */
	uptr->u5 &= ~(MT_MARK | MT_CMDMSK | MT_IDLE | MT_RDY | MT_MARK);
	r = sim_tape_sprecr(uptr, &reclen);
	mt_chan[chan] &= ~MTC_BSY;
	/* We don't set EOF on BSR */
	if (r == MTSE_TMK) {
	    sim_activate(uptr, 50);
	    return SCPE_OK;
	}
	sim_activate(uptr, 10 + (10 * reclen));
	break;

    case MT_BSF:
	uptr->u5 &= ~(MT_MARK | MT_IDLE | MT_RDY);
	r = sim_tape_sprecr(uptr, &reclen);
	/* If we hit mark or end of tape */
	if (r == MTSE_TMK || r == MTSE_BOT) {
	    sim_debug(DEBUG_DETAIL, dptr, "Backspace file unit=%d\n",
		      unit);
	    uptr->u5 &= ~MT_CMDMSK;
	    mt_chan[chan] &= ~MTC_BSY;
	    sim_activate(uptr, 50);
	} else {
	    sim_activate(uptr, 10 + (10 * reclen));
	}
	break;

    case MT_REW:
	sim_debug(DEBUG_DETAIL, dptr, "Rewind unit=%d\n", unit);
	uptr->u5 &= ~(MT_MARK | MT_CMDMSK | MT_IDLE | MT_RDY);
	r = sim_tape_rewind(uptr);
	sim_activate(uptr, 30000);
	mt_chan[chan] &= ~MTC_BSY;
	break;

    case MT_RUN:
	sim_debug(DEBUG_DETAIL, dptr, "Unload unit=%d\n", unit);
	uptr->u5 &= ~(MT_MARK | MT_CMDMSK | MT_IDLE | MT_RDY);
	r = sim_tape_detach(uptr);
	mt_chan[chan] &= ~MTC_BSY;
	break;

    }
    return mt_error(uptr, chan, r, dptr);
}

/* Boot from given device */
t_stat
mt_boot(int32 unit_num, DEVICE * dptr)
{
    UNIT               *uptr = &dptr->units[unit_num];
    int                 chan = UNIT_G_CHAN(uptr->flags);
    uint16              dev = unit_num + 020 + mt_dib.addr;
    t_stat              r;

    if ((uptr->flags & UNIT_ATT) == 0)
	return SCPE_UNATT;	/* attached? */

    /* Start a read. */
    if (mt(dptr->units, OP_RDS, dev) != 1) 
	return STOP_IONRDY;
    r = mt_read_buff(uptr, MT_RDSB, dptr);
    if (r != SCPE_OK)
	return r;

    /* Copy first three records. */
    M[0] = mt_buffer[chan][uptr->u6--];
    M[1] = mt_buffer[chan][uptr->u6--];
    if (chan != 0)
	M[2] = mt_buffer[chan][uptr->u6--];
    /* Make sure channel is set to start reading rest. */
    return chan_boot(unit_num, dptr);
}

void
mt_ini(UNIT * uptr, t_bool f)
{
    int                 chan = UNIT_G_CHAN(uptr->flags);

    if (uptr->flags & UNIT_ATT)
	uptr->u5 = MT_RDY;
    else
	uptr->u5 = 0;
    mt_chan[chan] = 0;
}

t_stat
mt_reset(DEVICE * dptr)
{
    return SCPE_OK;
}

t_stat
mt_attach(UNIT * uptr, char *file)
{
    t_stat              r;

    if ((r = sim_tape_attach(uptr, file)) != SCPE_OK)
	return r;
    uptr->u5 |= MT_RDY;
    uptr->flags |= MTUF_ONLINE;
    return SCPE_OK;
}

t_stat
mt_detach(UNIT * uptr)
{
    uptr->u5 = 0;
    uptr->flags &= ~MTUF_ONLINE;
    return sim_tape_detach(uptr);
}
