/* nova_pt.c: NOVA paper tape read/punch simulator

   Copyright (c) 1993, 1994, 1995,
   Robert M Supnik, Digital Equipment Corporation
   Commercial use prohibited

   ptr		paper tape reader
   ptp		paper tape punch
*/

#include "nova_defs.h"

extern int int_req, dev_busy, dev_done, dev_disable;
int ptr_stopioe = 0, ptp_stopioe = 0;			/* stop on error */
int ptr_svc (UNIT *uptr);
int ptp_svc (UNIT *uptr);
int ptr_reset (DEVICE *dptr);
int ptp_reset (DEVICE *dptr);
extern int sim_activate (UNIT *uptr, int interval);
extern int sim_cancel (UNIT *uptr);

/* PTR data structures

   ptr_dev	PTR device descriptor
   ptr_unit	PTR unit descriptor
   ptr_reg	PTR register list
*/

UNIT ptr_unit = {
	UDATA (&ptr_svc, UNIT_SEQ+UNIT_ATTABLE, 0), SERIAL_IN_WAIT };

REG ptr_reg[] = {
	{ ORDATA (BUF, ptr_unit.buf, 8) },
	{ FLDATA (BUSY, dev_busy, INT_V_PTR) },
	{ FLDATA (DONE, dev_done, INT_V_PTR) },
	{ FLDATA (DISABLE, dev_disable, INT_V_PTR) },
	{ FLDATA (INT, int_req, INT_V_PTR) },
	{ DRDATA (POS, ptr_unit.pos, 32), PV_LEFT },
	{ DRDATA (TIME, ptr_unit.wait, 24), PV_LEFT },
	{ FLDATA (STOP_IOE, ptr_stopioe, 0) },
	{ NULL }  };

DEVICE ptr_dev = {
	"PTR", &ptr_unit, ptr_reg, NULL,
	1, 10, 32, 1, 8, 8,
	NULL, NULL, &ptr_reset,
	NULL, NULL, NULL };

/* PTP data structures

   ptp_dev	PTP device descriptor
   ptp_unit	PTP unit descriptor
   ptp_reg	PTP register list
*/

UNIT ptp_unit = {
	UDATA (&ptp_svc, UNIT_SEQ+UNIT_ATTABLE, 0), SERIAL_OUT_WAIT };

REG ptp_reg[] = {
	{ ORDATA (BUF, ptp_unit.buf, 8) },
	{ FLDATA (BUSY, dev_busy, INT_V_PTP) },
	{ FLDATA (DONE, dev_done, INT_V_PTP) },
	{ FLDATA (DISABLE, dev_disable, INT_V_PTP) },
	{ FLDATA (INT, int_req, INT_V_PTP) },
	{ DRDATA (POS, ptp_unit.pos, 32), PV_LEFT },
	{ DRDATA (TIME, ptp_unit.wait, 24), PV_LEFT },
	{ FLDATA (STOP_IOE, ptp_stopioe, 0) },
	{ NULL }  };

DEVICE ptp_dev = {
	"PTP", &ptp_unit, ptp_reg, NULL,
	1, 10, 32, 1, 8, 8,
	NULL, NULL, &ptp_reset,
	NULL, NULL, NULL };

/* Paper tape reader: IOT routine */

int ptr (int pulse, int code, int AC)
{
int iodata;

iodata = (code == ioDIA)? ptr_unit.buf & 0377: 0;
switch (pulse) {					/* decode IR<8:9> */
case iopS: 						/* start */
	dev_busy = dev_busy | INT_PTR;			/* set busy */
	dev_done = dev_done & ~INT_PTR;			/* clear done, int */
	int_req = int_req & ~INT_PTR;
	sim_activate (&ptr_unit, ptr_unit.wait);	/* activate unit */
	break;
case iopC:						/* clear */
	dev_busy = dev_busy & ~INT_PTR;			/* clear busy */
	dev_done = dev_done & ~INT_PTR;			/* clear done, int */
	int_req = int_req & ~INT_PTR;
	sim_cancel (&ptr_unit);				/* deactivate unit */
	break;  }					/* end switch */
return iodata;
}

/* Unit service */

int ptr_svc (UNIT *uptr)
{
int temp;

if ((ptr_unit.flags & UNIT_ATT) == 0)			/* attached? */
	return IORETURN (ptr_stopioe, SCPE_UNATT);
if ((temp = getc (ptr_unit.fileref)) == EOF) {		/* end of file? */
	if (feof (ptr_unit.fileref)) {
		if (ptr_stopioe) printf ("PTR end of file\n");
		else return SCPE_OK;  }
	else perror ("PTR I/O error");
	clearerr (ptr_unit.fileref);
	return SCPE_IOERR;  }
dev_busy = dev_busy & ~INT_PTR;				/* clear busy */
dev_done = dev_done | INT_PTR;				/* set done */
int_req = (int_req & ~INT_DEV) | (dev_done & ~dev_disable);
ptr_unit.buf = temp & 0377;
ptr_unit.pos = ptr_unit.pos + 1;
return SCPE_OK;
}

/* Reset routine */

int ptr_reset (DEVICE *dptr)
{
ptr_unit.buf = 0;
dev_busy = dev_busy & ~INT_PTR;				/* clear busy */
dev_done = dev_done & ~INT_PTR;				/* clear done, int */
int_req = int_req & ~INT_PTR;
sim_cancel (&ptr_unit);					/* deactivate unit */
return SCPE_OK;
}

/* Paper tape punch: IOT routine */

int ptp (int pulse, int code, int AC)
{
if (code == ioDOA) ptp_unit.buf = AC & 0377;
switch (pulse) {					/* decode IR<8:9> */
case iopS: 						/* start */
	dev_busy = dev_busy | INT_PTP;			/* set busy */
	dev_done = dev_done & ~INT_PTP;			/* clear done, int */
	int_req = int_req & ~INT_PTP;
	sim_activate (&ptp_unit, ptp_unit.wait);	/* activate unit */
	break;
case iopC:						/* clear */
	dev_busy = dev_busy & ~INT_PTP;			/* clear busy */
	dev_done = dev_done & ~INT_PTP;			/* clear done, int */
	int_req = int_req & ~INT_PTP;
	sim_cancel (&ptp_unit);				/* deactivate unit */
	break;  }					/* end switch */
return 0;
}

/* Unit service */

int ptp_svc (UNIT *uptr)
{
dev_busy = dev_busy & ~INT_PTP;				/* clear busy */
dev_done = dev_done | INT_PTP;				/* set done */
int_req = (int_req & ~INT_DEV) | (dev_done & ~dev_disable);
if ((ptp_unit.flags & UNIT_ATT) == 0)			/* attached? */
	return IORETURN (ptp_stopioe, SCPE_UNATT);
if (putc (ptp_unit.buf, ptp_unit.fileref) == EOF) {
	perror ("PTP I/O error");
	clearerr (ptp_unit.fileref);
	return SCPE_IOERR;  }
ptp_unit.pos = ptp_unit.pos + 1;
return SCPE_OK;
}

/* Reset routine */

int ptp_reset (DEVICE *dptr)
{
ptp_unit.buf = 0;
dev_busy = dev_busy & ~INT_PTP;				/* clear busy */
dev_done = dev_done & ~INT_PTP;				/* clear done, int */
int_req = int_req & ~INT_PTP;
sim_cancel (&ptp_unit);					/* deactivate unit */
return SCPE_OK;
}
